unit PButV103;

{Unit for graphical button interface}

interface  {start interface section}

const
     LEFT_BUTTON = 0;   {Symbolic constant used for the left button}
     MIDDLE_BUTTON = 2; {Symbolic constant used for the middle button}
     RIGHT_BUTTON = 1;  {Symbolic constant used for the right button}

type
    _PBut_ListPtr = ^_PBut_ListType; {Linked list pointer type def.}
    _PBut_ListType = record          {Type def. used for the linked list data
                                      structure.  It stores the list of the
                                      buttons.}
                     name, key : string;  {"name" contains the label of the
                                           button, and "key" is the string
                                           used to reference the button.  This
                                           string is used in the TrapClick
                                           function.}
                     tfont, tsize, tcol : word; {Stores the font, size, and
                                                 color of the button label.}
                     col : word;  {Stores the color of the button}
                     xs, ys, xe, ye : integer; {xs = upper left x coordinate.
                                                ys = upper left y coordinate.
                                                xe = lower right x coordinate.
                                                ye = lower right y coordinate.}
                     down : boolean;  {Stores whether or not the button is
                                       currently depressed.}
                     prev, next : _PBut_ListPtr {Pointers used to link the
                                                 list together.}
                     end; {record}

    ButtonObject = object  {Type def. of Button Object}

                   public {Start of public sector}

                   procedure Init;
                      {Initializes the button object.}
                   procedure AddButton(xs, ys, xe, ye : integer; col : word; name, key : string);
                      {Adds a button to the button object, and the display list}
                   function RemoveButton(key : string) : boolean;
                      {Removes a button from the button object display list.  Returns false if there was an error.}
                   procedure ShowAllButtons;
                      {Displays all buttons on screen}
                   function TrapClick(ButNum : integer) : string;
                      {Waits until the user chooses a button.  It returns the key of the chosen button.}
                   procedure SetShadowLength(s : integer);
                      {Sets the shadow length of the button, in pixels.  The default length is 2 pixels.}
                   procedure SetRadiusLength(r : integer);
                      {Sets the radius length of the rounded corner, in pixels.  Default = 2}
                   procedure SetTextColor(col : word);
                      {Sets the current text color to the passed value.}
                   procedure SetTextFont(font : word);
                      {Sets the current text font to the passed value.}
                   procedure SetTextSize(size : word);
                      {Sets the current text size to the passed value.}
                   procedure Destruct;
                      {Cleans up after the button unit.  It frees all memory of the linked list.}

                   private  {Start of private sector}

                   ButtonList : _PBut_ListPtr; {private linked list data structure.  It stores all the buttons.
                                                It is private so that it is not accidentally modified.}
                   S_L : integer; {Private var. for the shadow length}
                   R_L : integer; {Private var. for the radius length}
                   _tfont, _tsize, _tcol : word; {_tfont = current text font
                                                  _tsize = current text size
                                                  _tcol = current text color}

                   end; {object}

implementation  {Start implementation section}

uses
    graph,  {Links in the graphics unit.  It is used to draw the buttons.}
    DOS;    {Links in the DOS unit.  It is used for BIOS interrupts to trap the mouse.}

procedure ButtonObject.Init;

begin {procedure}
     new(ButtonList);    {Allocates memory for the button linked list.}
     with ButtonList^ do {Places a header at the top of the list.}
          begin {with}
               xs := -1; {Flag values for header.}
               ys := -1; {          |            }
               xe := -1; {          |            }
               ye := -1; {          v            }
               next := ButtonList; {Links up the list}
               prev := ButtonList; {LInks up the list}
               col := 10000; {Flag value for header.}
               key := ' '; {Flag value for header.}
               down := false {The button doesn't exist, so it's not down}
          end; {with}
     S_L := 2; {Sets private var. shadow length to 2 pix.}
     R_L := 2; {Sets private var. shadow length to 2 pix.}
     _tcol := black; {Sets private var. text color to black.}
     _tsize := 3; {Sets private var. text size to 3.}
     _tfont := SmallFont {Sets private var. text font to Small Font}
end; {procedure}

procedure ButtonObject.AddButton;

var
   cur1, cur2 : _PBut_ListPtr; {Pointer vars. used to traverse the list.}

begin {procedure}
     cur1 := ButtonList; {Sets one pointer to the header.}
     while cur1^.next <> ButtonList do {Moves to the end of the list.}
           cur1 := cur1^.next; {Moves to the next record.}
     new(cur2); {Allocates memory for the other pointer.}
     cur2^.prev := cur1; {Links up the list.}
     cur1^.next := cur2; {Links up the list.}
     cur2^.xs := xs;     {Sets upper left x to the passed value}
     cur2^.ys := ys;     {Sets upper left y to the passed value}
     cur2^.xe := xe;     {Sets lower left x to the passed value}
     cur2^.ye := ye;     {Sets lower left y to the passed value}
     cur2^.name := name; {Sets button label to the passed value}
     cur2^.key := key;   {Sets button key to the passed value}
     cur2^.col := col;   {Sets button color to the passed value}
     cur2^.tfont := _tfont; {Sets button font to the private var. for it}
     cur2^.tsize := _tsize; {Sets button size to the private var. for it}
     cur2^.tcol := _tcol;   {Sets button color to the private var. for it}
     cur2^.down := false;   {The button is not yet depressed, so this is false}
     cur2^.next := ButtonList; {Links up the list.}
     ButtonList^.prev := cur2 {Links up the list.}
end; {procedure}

function ButtonObject.RemoveButton;

var
   cur, del : _PBut_ListPtr; {Pointer vars. used to delete an item.}

begin {function}
     cur := ButtonList^.next; {Sets one pointer to the record after the header}
     while (cur^.key <> key) and (cur <> ButtonList) do {Looks for a matching key}
           cur := cur^.next; {Moves to the next record}
     if cur^.key = key then {Checks to see if the key has been found}
        begin {if}
             RemoveButton := true; {No error!}
             del := cur; {Sets the deletion pointer to the other pointer.}
             cur := cur^.prev; {Moves the other pointer to the previous record.}
             cur^.next := cur^.next^.next; {Links up the list.}
             cur^.next^.prev := cur; {Links up the list.}
             dispose(del) {De-allocates the memory for the deleted record.}
        end {if}
     else {Key has not been found!}
         RemoveButton := false {ERROR!}
end; {function}

procedure ButtonObject.ShowAllButtons;

var
   cur : _PBut_ListPtr; {Pointer var. used to traverse the list.}

   procedure DrawButton(BDat : _PBut_ListType; On : boolean);
      {Draws a button}

   begin {procedure in a procedure}
        if not(On) then {The button is not depressed}
           begin {if}

                 {This section is rather self explanatory.  It draws a series
                  of rectangles and sections of circles and words to draw the
                  button.  Key:

                           PieSlice : Section of a circle.
                           Bar      : Filled in rectangle.

                  For more information, consult the Turbo Pascal 7 manual.}

                BDat.xs := BDat.xs + S_L;
                BDat.xe := BDat.xe + S_L;
                BDat.ys := BDat.ys + S_L;
                BDat.ye := BDat.ye + S_L;
                setfillstyle(SolidFill, BDat.col);
                setcolor(BDat.col);
                PieSlice(BDat.xs + R_L, BDat.ys + R_L, 90, 180, R_L);
                PieSlice(BDat.xs + R_L, BDat.ye - R_L, 180, 270, R_L);
                PieSlice(BDat.xe - R_L, BDat.ys + R_L, 0, 90, R_L);
                PieSlice(BDat.xe - R_L, BDat.ye - R_L, 270, 360, R_L);
                bar(BDat.xs, BDat.ys + R_L, BDat.xe, BDat.ye - R_L);
                bar(BDat.xs + R_L, BDat.ys, BDat.xe - R_L, BDat.ye);
                BDat.xs := BDat.xs - S_L;
                BDat.xe := BDat.xe - S_L;
                BDat.ys := BDat.ys - S_L;
                BDat.ye := BDat.ye - S_L;
                setfillstyle(SolidFill, BDat.col + 8);
                setcolor(BDat.col + 8);
                PieSlice(BDat.xs + R_L, BDat.ys + R_L, 90, 180, R_L);
                PieSlice(BDat.xs + R_L, BDat.ye - R_L, 180, 270, R_L);
                PieSlice(BDat.xe - R_L, BDat.ys + R_L, 0, 90, R_L);
                PieSlice(BDat.xe - R_L, BDat.ye - R_L, 270, 360, R_L);
                bar(BDat.xs, BDat.ys + R_L, BDat.xe, BDat.ye - R_L);
                bar(BDat.xs + R_L, BDat.ys, BDat.xe - R_L, BDat.ye);
                settextjustify(CenterText, CenterText);
                settextstyle(BDat.TFont, HorizDir, BDat.TSize);
                setcolor(BDat.tcol);
                OutTextXY((BDat.xs + BDat.xe) div 2, (BDat.ys + BDat.ye) div 2, BDat.name)
           end {if}
        else begin

                 {This section is rather self explanatory.  It draws a series
                  of rectangles and sections of circles and words to draw the
                  button.  Key:

                           PieSlice : Section of a circle.
                           Bar      : Filled in rectangle.

                  For more information, consult the Turbo Pascal 7 manual.}

             BDat.xs := BDat.xs;
             BDat.xe := BDat.xe;
             BDat.ys := BDat.ys;
             BDat.ye := BDat.ye;
             setfillstyle(SolidFill, Black);
             SetColor(Black);
             PieSlice(BDat.xs + R_L, BDat.ys + R_L, 90, 180, R_L);
             PieSlice(BDat.xs + R_L, BDat.ye - R_L, 180, 270, R_L);
             PieSlice(BDat.xe - R_L, BDat.ys + R_L, 0, 90, R_L);
             PieSlice(BDat.xe - R_L, BDat.ye - R_L, 270, 360, R_L);
             bar(BDat.xs, BDat.ys + R_L, BDat.xe, BDat.ye - R_L);
             bar(BDat.xs + R_L, BDat.ys, BDat.xe - R_L, BDat.ye);
             BDat.xs := BDat.xs + S_L;
             BDat.xe := BDat.xe + S_L;
             BDat.ys := BDat.ys + S_L;
             BDat.ye := BDat.ye + S_L;
             setfillstyle(SolidFill, BDat.col + 8);
             setcolor(BDat.col + 8);
             PieSlice(BDat.xs + R_L, BDat.ys + R_L, 90, 180, R_L);
             PieSlice(BDat.xs + R_L, BDat.ye - R_L, 180, 270, R_L);
             PieSlice(BDat.xe - R_L, BDat.ys + R_L, 0, 90, R_L);
             PieSlice(BDat.xe - R_L, BDat.ye - R_L, 270, 360, R_L);
             bar(BDat.xs, BDat.ys + R_L, BDat.xe, BDat.ye - R_L);
             bar(BDat.xs + R_L, BDat.ys, BDat.xe - R_L, BDat.ye);
             settextjustify(CenterText, CenterText);
             settextstyle(BDat.TFont, HorizDir, BDat.TSize);
             setcolor(BDat.tcol);
             OutTextXY((BDat.xs + BDat.xe) div 2 + S_L, (BDat.ys + BDat.ye) div 2 + S_L, BDat.name)
        end {else}
   end; {procedure in a procedure}

begin {procedure}
     cur := ButtonList^.next; {Sets the pointer to the record after the header.}
     while cur <> ButtonList do {Draws all the buttons on screen}
           begin {with}
                DrawButton(cur^, false); {Draws the button.  "false" means that it isn't depressed}
                cur := cur^.next {Moves the pointer to the next record}
           end {with}
end; {procedure}

function ButtonObject.TrapClick;

var
   xpos, ypos : integer; {position counters}
   cur : _PBut_ListPtr;  {pointer var.}
   regs : registers;     {DOS registers}
   picked : boolean;     {Boolean flag which represents whether or not a button has been selected.}

   function ButtonPressed(BNum : integer; BData : integer) : boolean;

   {Checks to see if the button requested is depressed}

   begin {function}
        case BNum of {Checks to see which button needs to be analyzed}
             LEFT_BUTTON : ButtonPressed := BData mod 2 = 1; {Checks with DOS register data.}
             RIGHT_BUTTON : ButtonPressed := BData div 2 mod 2 = 1; {Checks with DOS register data.}
             MIDDLE_BUTTON : ButtonPressed := BData div 4 = 1 {Checks with DOS register data.}
        end {case}
   end; {function}

   procedure DrawButton(BDat : _PBut_ListType; On : boolean);
      {Draws a button.  It is identical to the one in the previous procedure,
      and so will not be explained here.  Look at the previous one for details.}

   begin
        if not(On) then
           begin
                BDat.xs := BDat.xs + S_L;
                BDat.xe := BDat.xe + S_L;
                BDat.ys := BDat.ys + S_L;
                BDat.ye := BDat.ye + S_L;
                setfillstyle(SolidFill, BDat.col);
                setcolor(BDat.col);
                PieSlice(BDat.xs + R_L, BDat.ys + R_L, 90, 180, R_L);
                PieSlice(BDat.xs + R_L, BDat.ye - R_L, 180, 270, R_L);
                PieSlice(BDat.xe - R_L, BDat.ys + R_L, 0, 90, R_L);
                PieSlice(BDat.xe - R_L, BDat.ye - R_L, 270, 360, R_L);
                bar(BDat.xs, BDat.ys + R_L, BDat.xe, BDat.ye - R_L);
                bar(BDat.xs + R_L, BDat.ys, BDat.xe - R_L, BDat.ye);
                BDat.xs := BDat.xs - S_L;
                BDat.xe := BDat.xe - S_L;
                BDat.ys := BDat.ys - S_L;
                BDat.ye := BDat.ye - S_L;
                setfillstyle(SolidFill, BDat.col + 8);
                setcolor(BDat.col + 8);
                PieSlice(BDat.xs + R_L, BDat.ys + R_L, 90, 180, R_L);
                PieSlice(BDat.xs + R_L, BDat.ye - R_L, 180, 270, R_L);
                PieSlice(BDat.xe - R_L, BDat.ys + R_L, 0, 90, R_L);
                PieSlice(BDat.xe - R_L, BDat.ye - R_L, 270, 360, R_L);
                bar(BDat.xs, BDat.ys + R_L, BDat.xe, BDat.ye - R_L);
                bar(BDat.xs + R_L, BDat.ys, BDat.xe - R_L, BDat.ye);
                settextjustify(CenterText, CenterText);
                settextstyle(BDat.TFont, HorizDir, BDat.TSize);
                setcolor(BDat.tcol);
                OutTextXY((BDat.xs + BDat.xe) div 2, (BDat.ys + BDat.ye) div 2, BDat.name)
           end
        else begin
             BDat.xs := BDat.xs;
             BDat.xe := BDat.xe;
             BDat.ys := BDat.ys;
             BDat.ye := BDat.ye;
             setfillstyle(SolidFill, Black);
             setcolor(Black);
             PieSlice(BDat.xs + R_L, BDat.ys + R_L, 90, 180, R_L);
             PieSlice(BDat.xs + R_L, BDat.ye - R_L, 180, 270, R_L);
             PieSlice(BDat.xe - R_L, BDat.ys + R_L, 0, 90, R_L);
             PieSlice(BDat.xe - R_L, BDat.ye - R_L, 270, 360, R_L);
             bar(BDat.xs, BDat.ys + R_L, BDat.xe, BDat.ye - R_L);
             bar(BDat.xs + R_L, BDat.ys, BDat.xe - R_L, BDat.ye);
             BDat.xs := BDat.xs + S_L;
             BDat.xe := BDat.xe + S_L;
             BDat.ys := BDat.ys + S_L;
             BDat.ye := BDat.ye + S_L;
             setfillstyle(SolidFill, BDat.col + 8);
             setcolor(BDat.col + 8);
             PieSlice(BDat.xs + R_L, BDat.ys + R_L, 90, 180, R_L);
             PieSlice(BDat.xs + R_L, BDat.ye - R_L, 180, 270, R_L);
             PieSlice(BDat.xe - R_L, BDat.ys + R_L, 0, 90, R_L);
             PieSlice(BDat.xe - R_L, BDat.ye - R_L, 270, 360, R_L);
             bar(BDat.xs, BDat.ys + R_L, BDat.xe, BDat.ye - R_L);
             bar(BDat.xs + R_L, BDat.ys, BDat.xe - R_L, BDat.ye);
             settextjustify(CenterText, CenterText);
             settextstyle(BDat.TFont, HorizDir, BDat.TSize);
             setcolor(BDat.tcol);
             OutTextXY((BDat.xs + BDat.xe) div 2 + S_L div 2, (BDat.ys + BDat.ye) div 2 + S_L div 2, BDat.name)
        end
   end;

   function OnButton(x, y : integer; BDat : _PBut_ListType) : boolean;
      {Checks to see if the click was on the button requested}

   begin {function in a function}
         {This IF segment just checks to see if the click was in button bounds}
        if (x >= BDat.xs) and (x <= BDat.xe) and (y >= BDat.ys) and (y <= BDat.ye) then
           begin {if}
                if (x >= BDat.xs) and (y >= BDat.ys) and (x <= BDat.xs + R_L) and (y <= BDat.ys + R_L) then
                   if sqrt(sqr(x - BDat.xs) + sqr(y - BDat.ys)) <= R_L then
                      OnButton := true
                   else
                       OnButton := false
                else if (x >= BDat.xs) and (y <= BDat.ye) and (x <= BDat.xs - R_L) and (y >= BDat.ye - R_L) then
                     if sqrt(sqr(x - BDat.xs) + sqr(y - BDat.ye)) <= R_L then
                        OnButton := true
                     else
                         OnButton := false
                else if (x <= BDat.xe) and (x >= BDat.xe - R_L) and (y >= BDat.ys) and (y <= BDat.ys + R_L) then
                     if sqrt(sqr(x - BDat.xe) + sqr(y - BDat.ys)) <= R_L then
                        OnButton := true
                     else
                         OnButton := false
                else if (x <= BDat.xe) and (x >= BDat.xe - R_L) and (y <= BDat.ye) and (y >= BDat.ye - R_L) then
                     if sqrt(sqr(x - BDat.xe) + sqr(y - BDat.ye)) <= R_L then
                        OnButton := true
                     else
                         OnButton := false
                else
                    OnButton := true
           end {if}
        else
            OnButton := false
   end; {function in a function}


begin {function}
     xpos := -1; {Sets x position counter to -1 (Flag)}
     ypos := -1; {Sets y position counter to -1 (Flag)}
     picked := false; {No button has been selected yet...}

     repeat
           regs.ax := $0003;
           intr($33, regs)
     until ButtonPressed(ButNum, regs.bx);
     repeat {Waits until a button is selected}
           regs.ax := $0005;
           regs.bx := ButNum; {Interrupt for button status}
           intr($33, regs);
           xpos := regs.cx; {Gets x position of last click}
           ypos := regs.dx; {Gets y position of last click}
           cur := ButtonList^.next; {Moves the pointer to the record after the header.}
           while (cur <> ButtonList) and not(picked) do {Searches through all the existing buttons}
                 begin {while}
                      if OnButton(xpos, ypos, cur^) then {If the last click was on a button...}
                         begin {if}
                              regs.ax := $0003; {Checks to see if the button is depressed now}
                              intr($33, regs);
                              if OnButton(regs.cx, regs.dx, cur^) and ButtonPressed(ButNum, regs.bx) and not(cur^.down) then
                                 {If the button is being clicked on now.}
                                 begin {if}
                                      regs.ax := $0002; {Hides mouse cursor}
                                      intr($33, regs);
                                      DrawButton(cur^, true); {Draws depressed button}
                                      regs.ax := $0001;{Shows mouse cursor}
                                      intr($33, regs);
                                      cur^.down := true {The button is down}
                                 end {if}
                              else if not(OnButton(regs.cx, regs.dx, cur^) and ButtonPressed(ButNum, regs.bx)) and
                                      cur^.down then
                                         {if the button is down, but it isn't being clicked on, then it is displayed up.}
                                         begin {if}
                                              regs.ax := $0002; {Hide mouse cursor}
                                              intr($33, regs);
                                              DrawButton(cur^, false); {Draws normal button}
                                              regs.ax := $0001; {Show mouse cursor}
                                              intr($33, regs);
                                              cur^.down := false {The button is not down}
                                         end; {if}
                              regs.ax := $0006;  {Checks specific button}
                              regs.bx := ButNum; {Checks for the passed button}
                              intr($33, regs);
                              if OnButton(regs.cx, regs.dx, cur^) then {If the button has been selected...}
                                 begin {if}
                                      regs.ax := $0003; {Check current button condition}
                                      intr($33, regs);
                                      case ButNum of {Checks individual button status}
                                           LEFT_BUTTON : if regs.bx mod 2 = 0 then {It has been picked!}
                                                            begin {if}
                                                                 picked := true; {Picked!}
                                                                 TrapClick := cur^.key {Returns the valued of the key}
                                                            end; {if}
                                           RIGHT_BUTTON : if regs.bx div 2 mod 2 = 0 then {It has been picked!}
                                                             begin {if}
                                                                  picked := true; {Picked!}
                                                                  TrapClick := cur^.key {Returns the valued of the key}
                                                             end; {if}
                                           MIDDLE_BUTTON : if regs.bx div 4 = 1 then {It has been picked!}
                                                              begin
                                                                   picked := true; {Picked!}
                                                                   TrapClick := cur^.key {Returns the valued of the key}
                                                              end {if}
                                      end {case}
                                 end {if}
                         end {if}
                      else begin {it hasn't been picked}
                           if cur^.down then {If the button is depressed...}
                              begin {if}
                                   regs.ax := $0002; {Hide mouse}
                                   intr($33, regs);
                                   DrawButton(cur^, false); {Draws normal button}
                                   regs.ax := $0001; {Show mouse}
                                   intr($33, regs);
                                   cur^.down := false {The button is NOT down}
                              end {if}
                      end; {else}
                      cur := cur^.next {moves pointer to next record}
                 end {while}
     until picked {repeats until a button has been selected}
end; {function}

procedure ButtonObject.SetShadowLength;

begin {procedure}
     S_L := s {Sets private var. to passed value}
end; {procedure}

procedure ButtonObject.SetRadiusLength;

begin {procedure}
     R_L := r {Sets private var. to passed value}
end; {procedure}

procedure ButtonObject.SetTextFont;

begin {procedure}
     _tfont := font {Sets private var. to passed value}
end; {procedure}

procedure ButtonObject.SetTextSize;

begin {procedure}
     _tsize := size {Sets private var. to passed value}
end; {procedure}

procedure ButtonObject.SetTextColor;

begin {procedure}
     _tcol := col {Sets private var. to passed value}
end; {procedure}

procedure ButtonObject.Destruct;

var
   del, cur : _PBut_ListPtr;  {Pointers used for deletion}

begin {procedure}
     cur := ButtonList^.prev; {Moves pointer to header's predecessor}
     while cur <> ButtonList do {De-allocates all memory used for list}
           begin {while}
                del := cur; {Sets deletion pointer to the other pointer}
                cur := cur^.prev; {Moves other pointer to previous record}
                cur^.next := cur^.next^.next; {Links up the list}
                cur^.next^.prev := cur; {Links up the list}
                dispose(del) {De-allocates memory for data at deletion pointer}
           end; {while}
     dispose(ButtonList) {De-allocates memory for header}
end; {procedure}

end. {unit}