unit ALIDRAW;

interface

uses CRT, {Links in screen unit.  Enables calls for screen clearing and color
           changing}
     DOS, {Links in DOS unit.  Used for time functions}
     graph, {Links in BGI graphics pack.  Used for graphics}
     PFontV13, {Links in Po-Shen's font library and animation kit.  Also
                enables graphical reading.}
     PButV103, {Links in Po-Shen's old button library.}
     PButV104, {Links in Po-Shen's button library.  It allows easy buttons
                by implementing them inside objects, in the private sector.}
     PMseV101, {Links in Po-Shen's mouse library.  It enables mouse calls.}
     drivers, {Links in the drivers unit.  Allows trapping of CTRL, ALT,
               SHIFT, etc.}
     PSndV102; {Links in Po-Shen's sound blaster collection, created
               mainly by Ethan Brodsky.  Sound file copy protection
               provided by Po-Shen Loh.}

const
     XMSRequired   = 900; {XMS memory required to load the sounds (KBytes)}
     SharedEMB = false; {Do not share the extended memory block (XMS)}
     NumSounds = 9; {Number of sounds used in this program.}
     SoundPaths : array[1..3] of string = ('', 'E:\POSHEN\PASCAL\CURRENT\',
                                           '..\SNDFILES\'); {Possible paths of sound files}
     GraphPaths : array[1..5] of string = ('', 'E:\POSHEN\PASCAL\CURRENT\',
                                           '..\GPHFILES\',
                                           'E:\POSHEN\PASCAL\SOURCE\GAME\',
                                           'E:\POSHEN\PASCAL\PROJECT\DOWN\'); {Possible paths of graphics files}
     IniPaths : array[1..3] of string = ('', 'E:\POSHEN\PASCAL\PROJECT\',
                                         'J:\OLD\PROJECT\'); {Possible paths of .ini files}
     IniFileName = 'ALIATT.INI'; {Name of ini file}
     Fill50 : FillPatternType = ($AA, $55, $AA, $55, $AA, $55, $AA, $55); {User-defined fill pattern.  50% fill}
     FlashSize = 8; {Size of flashing text}
     FlashSpeed = 30; {Speed of flashing text}

     MenuBorderColor = Green; {Color of menu border}
     MenuTextColor = LightGreen; {Color of button text}
     MenuButtonColor = LightGreen; {Color of buttons}

     MaxGameY = 459; {Maximum Y coord. during game play}

     NumWeapons = 9; {Number of weapons}
     NumSpecials = 3; {Number of special weapons}

     HEADER = 0; {Reference constants}

     LARGE = 3;
     MEDIUM = 2;
     SMALL = 1;

     BULLET = 0;
     POWERGUN_BULLET = 1;
     LASERPULSE_PULSE = 2;
     SPE_OFFSET = 128;
     AAMINE = 1;
     SUPERSHIELD = 2;
     MEGAMISSILE = 3;

     INVULNERABLE_TIME = 1500; {Time invulnerable if super-shield activated}
     LIFE_POINTS = 2000; {Points to get a new life}
     DEAD_INVULNERABLE = 500; {5 invulnerable seconds if you are dead}
     WARNING_TIME = 200; {Warning time before invulnerability wears out}

     CASH = 0; {Reference constants}
     SHIELD = 1;

     GUNS = 0;
     THRUSTERS = 1;
     SHIELDS = 2;
     GENERATORS = 3;

     BasicGenerator = 1;
     BasicThruster = 1;
     LEV2THRUSTER = 2;

     RECHARGE_AMT = 2; {Game play constants}
     ACC_PWR = 3;
     Y_Move_Delay = 5;
     Y_MOVE_COEFF = 1;
     Y_CANNON_REFIRE_DELAY = 40;
     Y_AUTOMATIC_DELAY = 10;
     Y_POWERGUN_REFIRE_DELAY = 100;
     Y_LASERPULSE_REFIRE_DELAY = 100;
     Y_SPRAY_REFIRE_DELAY = 60;
     Y_MAX_VELOCITY = 10;
     Y_ACC_INCREMENT = 1;
     Y_TURN_DELAY = 1;
     Y_TURN_ANGLE = Pi/30;
     BULLET_SPEED = 10;
     BULLET_RANGE = 30;
     BULLET_POWER = 5;
     BULLET_TRAVEL_DELAY = 2;
     POWERGUN_POWER = 5;
     POWERGUN_BULLET_RANGE = 10;
     LASERPULSE_SPEED = 10;
     LASERPULSE_PULSE_RANGE = 40;
     LASERPULSE_PULSE_LENGTH = 10;
     LASERPULSE_POWER = 40;
     LASERPULSE_TRAVEL_DELAY = 2;
     SPRAY_POWER = 5;
     SPECIAL_VELOCITY = 2;
     E_BULLET_SPEED = 10;
     E_BULLET_RANGE = 10;
     RECOIL_INCREMENT = -0.01;
     SWITCH_DELAY = 20;
     AST_MOVE_DELAY = 10;
     AST_RADIUS = 5;
     AST_DAMAGE = 2;
     SPECIAL_REFIRE_DELAY = 20;
     AAMINE_MOVE_DELAY = 5;
     AA_MINE_RADIUS = 5;
     HOMING_VELOCITY = 3;
     AA_MINE_MAX_VELOCITY = 10;
     MEGAMISSILE_TRAVEL_DELAY = 2;
     MEGAMISSILE_LENGTH = 8;
     MEGAMISSILE_VELOCITY = 12;
     SPLIT_DIST = 10;
     MEGAMISSILE_RANGE = 40;
     MMEXPLODE_RANGE = 5;
     MSPLIT_ANGLE = Pi/18;
     MSPLIT_NUM = 3;
     ENEMY_RADIUS = 10;
     GUNSHIP_1 = 0;
     GUNSHIP_1_MOVE_DELAY = 5;
     GUNSHIP_1_SHOOT_DELAY = 80;
     GUNSHIP_1_RADIUS = 10;
     SHIELD_SUCKER = 1;
     SHIELD_SUCKER_MOVE_DELAY = 6;
     SHIELD_SUCKER_SHOOT_DELAY = 500;
     SHIELD_SUCKER_RADIUS = 10;
     SHIELD_SUCK_AMT = 1;
     CASH_MOVE_DELAY = 10;
     CASH_RADIUS = 3;
     CASH_VALUE = 1;
     SHIELD_MOVE_DELAY = 10;
     SHIELD_RADIUS = 3;
     SHIELD_VALUE = 1;
     GOODY_STAY_TIME = 3000;
     STAR_RADIUS = 1;
     STAR_MOVE_DELAY = 2;

     NumStarsOnScreen = 50;

     SPRAY_ANGLE = Pi/18;

     HALO_OUTER = 20;
     HALO_INNER = 12;
     INV_HALO_OUTER = 14;

     GoodChance = 2;
     NumExplode = 4;

     KbdInt = $15; {Interrupt to link keyboard handler to}

     _UP =     $48; {Reference constants}
     _DOWN =   $50;
     _LEFT =   $4B;
     _RIGHT =  $4D;
     _CTRL =   $1D;
     _ESC =    $01;
     _LSHIFT = $2A;
     _RSHIFT = $36;
     _ALT =    $38;
     _TAB =    $0F;

     BULLET_SOUND = 2; {Sound constants}
     E_BULLET_SOUND = 4;
     EXP_SND = 3;
     HIT_NOISE = 6;
     MONEY_NOISE = 7;
     SHIELD_NOISE = 8;
     LASER_SOUND = 5;
     SUCK_NOISE = 9;
     SUPERSHIELD_SOUND = 8;

     CASH_POINT_VALUE = 10; {Point value constants}
     SHIELD_POINT_VALUE = 5;
     GUNSHIP_1_POINT_VALUE = 20;
     SHIELD_SUCKER_POINT_VALUE = 20;
     ROCK_POINT_VALUE = 5;

     Prices : array[GUNS..GENERATORS, 1..NumWeapons] of longint =
      ((1, 30, 20, 40, 80, 10, 15, 20, 25), (1, 4, 6, 8, 10, 12, 14, 16, 18),
       (1, 3, 6, 7, 9, 11, 13, 15, 17), (1, 3, 6, 9, 12, 15, 18, 21, 24));
       {Prices of upgrades}
     SpePrices : array[1..NumSpecials] of longint = (1, 10, 10); {Price of special weapon}

type
    Sounds = array[1..NumSounds] of PSound; {Sound array}

    SpeType = array[1..NumSpecials] of shortint; {Array of specials}
    Person = record {Record to load of player}
                   Lives : longint; {Lives left}
                   LastLifeUp : longint; {Last score at which extra life was given}
                   Gun : integer; {Weapon posessed}
                   Automatic : boolean; {Automatic gun}
                   Specials : SpeType; {Specials posessed}
                   SpePosition : integer; {Current special}
                   ShieldType : byte; {Shield system}
                   MaxShields : integer; {Maximum shield level}
                   GeneratorType : byte; {Generator type}
                   RechargeDelay : real; {Generation speed}
                   Thruster : byte; {Thruster type}
                   AccDelay : real; {Acceleration speed}
                   Level : longint; {Level (game)}
                   Money : longint; {Credits}
                   Points : longint; {Points}
                   Name : string {Pilot's name}
             end;
    Vector = record
                   theta : real;
                   mag : real
             end; {Record}
    Coord = record
                  X, Y : real
            end; {record}
    YShipType = record
                      Shields : integer;
                      Power : integer;
                      Bearing : Vector;
                      Position : Coord;
                      Aim : real;
                      mov : longint;
                      shoot : longint;
                      acc : longint;
                      turn : longint;
                      spec : longint;
                      specswitch : longint;
                      recharge : longint;
                      invulnerablestay : longint
                end; {record}
    EShipPtr = ^EShipType;
    EShipType = record
                      what : byte;
                      Shields : integer;
                      mov : longint;
                      shoot : longint;
                      weap : byte;
                      Bearing : Vector;
                      Position : Coord;
                      Aim : real;
                      brake : longint;
                      acc : longint;
                      turn : longint;
                      prev, next : EShipPtr
                end; {record}
    AstPtr = ^AstType;
    AstType = record
                    Size : integer;
                    Bearing : Vector;
                    Position : Coord;
                    move : longint;
                    prev, next : AstPtr
              end; {record}
    GoodPtr = ^GoodyType;
    GoodyType = record
                      what : byte;
                      Bearing : Vector;
                      Position : Coord;
                      move : longint;
                      stay : longint;
                      prev, next : GoodPtr
                end; {record}
    ProjPtr = ^Projtype;
    Projtype = record
                     what : byte;
                     Bearing : Vector;
                     Position : Coord;
                     move : longint;
                     sx, sy : integer;
                     range : integer;
                     prev, next : ProjPtr
               end; {record}
    StarPtr = ^StarItem;
    StarItem = record
                     move : longint;
                     XSpeed : integer;
                     YSpeed : integer;
                     Position : Coord;
                     prev, next : StarPtr
               end; {record}
    LevelType = record
                      NumAsts : longint;
                      AlienChance : longint;
                      AlienLevel : byte
                end; {record}

procedure EraseYShip(x, y, dir : real);
procedure DrawYShip(x, y, dir : real);
procedure EraseIYShip(x, y, dir : real);
procedure DrawIYShip(x, y, dir : real);
procedure DrawWYShip(x, y, dir : real);
procedure EraseBullet(x, y : real);
procedure DrawBullet(x, y : real);
procedure ErasePBullet(x, y : real);
procedure DrawPBullet(x, y : real);
procedure EraseLaserpulsePulse(x, y : real; Bearing : Vector);
procedure DrawLaserpulsePulse(x, y : real; Bearing : Vector);
procedure EraseAst(x, y : real; Size : byte);
procedure DrawAst(x, y : real; Size : byte);
procedure EraseGunship1(X, Y : real);
procedure DrawGunship1(X, Y : real);
procedure DrawCash(X, Y : real);
procedure EraseCash(X, Y : real);
procedure DrawShield(X, Y : real);
procedure EraseShield(X, Y : real);
procedure DrawAAMine(X, Y : real);
procedure EraseAAMine(X, Y : real);
procedure DrawMMissile(X, Y, theta : real);
procedure EraseMMissile(X, Y, theta : real);
procedure DrawHalo(X, Y : real; Color : word);
procedure EraseHalo(X, Y : real);
procedure DrawShieldMeter(Amt, Max : integer);
procedure EraseShieldMeter;
procedure DrawPower(Amt, Max : integer);
procedure ErasePower;
procedure EraseCashMeter(Money : longint);
procedure DrawCashMeter(Money : longint);
procedure EraseLifeMeter;
procedure DrawLifeMeter(Lives : integer);
procedure EraseScoreMeter(Score : longint);
procedure DrawScoreMeter(Score : Longint);
procedure EraseSpecialsMeter(Pilot : Person);
procedure DrawSpecialsMeter(Pilot : Person);
procedure DrawMeterBox;
procedure ShowPilot(Pilot : Person);
procedure ErasePilot;
procedure CheckForNewLife(var Pilot : Person);
procedure FixYou(var You : YShipType; var Pilot : Person);
procedure DrawGun(X, Y, GType : integer);
procedure EraseGun(X, Y, GType : integer);
procedure DrawThruster(X, Y, TType : integer);
procedure EraseThruster(X, Y, TType : integer);
procedure DrawShieldDesc(X, Y, SType : integer);
procedure EraseShieldDesc(X, Y, SType : integer);
procedure DrawGenerator(X, Y, GType : integer);
procedure EraseGenerator(X, Y, GType : integer);
procedure DrawSpec(X, Y, SType : integer);
procedure EraseSpec(X, Y, SType : integer);
procedure CheckShieldSuckerHit(var Projs : ProjPtr; var hitbullet : boolean; var cur : EShipPtr; var Pilot : Person;
                               var Sound : Sounds; NoSFX : boolean);
procedure DrawShieldSucker(X, Y : real);
procedure EraseShieldSucker(X, Y : real);
procedure MSplit(var Missile : ProjPtr);
procedure MMExplode(var Missile : ProjPtr);
procedure EraseStar(X, Y : real);
procedure DrawStar(X, Y : real);
procedure ShowCredits(var EscFlag : boolean);
procedure SetUp(var par : paratype);

implementation

procedure EraseYShip(x, y, dir : real);

begin
     setfillstyle(solidfill, black);
     setcolor(black);
     fillellipse(round(x), round(y), INV_HALO_OUTER, INV_HALO_OUTER)
end;

procedure DrawYShip(x, y, dir : real);

begin
     setcolor(LightGreen);
     line(round(x + 10 * cos(dir)), round(y + 10 * sin(dir)), round(x + 10 * cos(dir+2*Pi/3)), round(y+10 * sin(dir+2*Pi/3)));
     line(round(x + 10 * cos(dir + 2*Pi/3)), round(y + 10 * sin(dir + 2*Pi/3)), round(x), round(y));
     line(round(x), round(y), round(x + 10 * cos(dir + 4*Pi/3)), round(y + 10 * sin(dir + 4*Pi/3)));
     line(round(x+10 * cos(dir + 4*Pi/3)), round(y + 10 * sin(dir + 4*Pi/3)), round(x + 10 * cos(dir)), round(y+10 * sin(dir)))
end;

{*******************************************************************}

procedure EraseIYShip(x, y, dir : real);

begin
     setfillstyle(solidfill, black);
     setcolor(black);
     fillellipse(round(x), round(y), INV_HALO_OUTER, INV_HALO_OUTER)
end;

procedure DrawIYShip(x, y, dir : real);

begin
     setcolor(LightCyan);
     circle(round(x), round(y), INV_HALO_OUTER);
     setcolor(LightGreen);
     line(round(x + 10 * cos(dir)), round(y + 10 * sin(dir)), round(x + 10 * cos(dir+2*Pi/3)), round(y+10 * sin(dir+2*Pi/3)));
     line(round(x + 10 * cos(dir + 2*Pi/3)), round(y + 10 * sin(dir + 2*Pi/3)), round(x), round(y));
     line(round(x), round(y), round(x + 10 * cos(dir + 4*Pi/3)), round(y + 10 * sin(dir + 4*Pi/3)));
     line(round(x+10 * cos(dir + 4*Pi/3)), round(y + 10 * sin(dir + 4*Pi/3)), round(x + 10 * cos(dir)), round(y+10 * sin(dir)))
end;

procedure DrawWYShip(x, y, dir : real);

begin
     setcolor(Cyan);
     circle(round(x), round(y), INV_HALO_OUTER);
     setcolor(LightGreen);
     line(round(x + 10 * cos(dir)), round(y + 10 * sin(dir)), round(x + 10 * cos(dir+2*Pi/3)), round(y+10 * sin(dir+2*Pi/3)));
     line(round(x + 10 * cos(dir + 2*Pi/3)), round(y + 10 * sin(dir + 2*Pi/3)), round(x), round(y));
     line(round(x), round(y), round(x + 10 * cos(dir + 4*Pi/3)), round(y + 10 * sin(dir + 4*Pi/3)));
     line(round(x+10 * cos(dir + 4*Pi/3)), round(y + 10 * sin(dir + 4*Pi/3)), round(x + 10 * cos(dir)), round(y+10 * sin(dir)))
end;

{*******************************************************************}

procedure EraseBullet(x, y : real);

begin
     setfillstyle(solidfill, black);
     bar(round(x) - 1, round(y) - 1, round(x) + 1, round(y) + 1)
end;

procedure DrawBullet(x, y : real);

begin
     setcolor(LightRed);
     setfillstyle(solidfill, LightRed);
     fillellipse(round(x), round(y), 1, 1)
end;

{*******************************************************************}

procedure ErasePBullet(x, y : real);

begin
     setfillstyle(solidfill, black);
     bar(round(x) - 1, round(y) - 1, round(x) + 1, round(y) + 1)
end;

procedure DrawPBullet(x, y : real);

begin
     setcolor(LightCyan);
     setfillstyle(solidfill, LightCyan);
     fillellipse(round(x), round(y), 1, 1)
end;

{*******************************************************************}

procedure EraseLaserpulsePulse(x, y : real; Bearing : Vector);

begin
     setcolor(Black);
     line(round(x), round(y), round(x + cos(Bearing.theta) * LASERPULSE_PULSE_LENGTH),
          round(y + sin(Bearing.theta) * LASERPULSE_PULSE_LENGTH))
end;

procedure DrawLaserpulsePulse(x, y : real; Bearing : Vector);

begin
     setcolor(LightGreen);
     line(round(x), round(y), round(x + cos(Bearing.theta) * LASERPULSE_PULSE_LENGTH),
          round(y + sin(Bearing.theta) * LASERPULSE_PULSE_LENGTH))
end;

{*******************************************************************}

procedure EraseAst(x, y : real; Size : byte);

begin
     setcolor(black);
     setfillstyle(solidfill, black);
     fillellipse(round(x), round(y), Size * AST_RADIUS, Size * AST_RADIUS)
end;

procedure DrawAst(x, y : real; Size : byte);

begin
     setfillpattern(Fill50, lightgray);
     setcolor(darkgray);
     fillellipse(round(x), round(y), Size * AST_RADIUS, Size * AST_RADIUS)
end;

{*******************************************************************}

procedure EraseGunship1(X, Y : real);

begin
     setcolor(black);
     setfillstyle(solidfill, black);
     fillellipse(round(X), round(Y), GUNSHIP_1_RADIUS, GUNSHIP_1_RADIUS)
end;

procedure DrawGunship1(X, Y : real);

begin
     setcolor(lightcyan);
     setfillstyle(solidfill, lightcyan);
     fillellipse(round(X), round(Y), GUNSHIP_1_RADIUS, GUNSHIP_1_RADIUS)
end;

{*******************************************************************}

procedure DrawCash(X, Y : real);

begin
     setcolor(LightGreen);
     setfillstyle(solidfill, lightgreen);
     fillellipse(round(X), round(Y), CASH_RADIUS, CASH_RADIUS)
end;

procedure EraseCash(X, Y : real);

begin
     setcolor(black);
     setfillstyle(solidfill, black);
     fillellipse(round(X), round(Y), CASH_RADIUS, CASH_RADIUS)
end;

{*******************************************************************}

procedure DrawShield(X, Y : real);

begin
     setcolor(LightBlue);
     setfillstyle(solidfill, lightblue);
     fillellipse(round(X), round(Y), SHIELD_RADIUS, SHIELD_RADIUS)
end;

procedure EraseShield(X, Y : real);

begin
     setcolor(black);
     setfillstyle(solidfill, black);
     fillellipse(round(X), round(Y), SHIELD_RADIUS, SHIELD_RADIUS)
end;

{*******************************************************************}

procedure DrawAAMine(X, Y : real);

begin
     setcolor(LightMagenta);
     setfillstyle(solidfill, lightblue);
     fillellipse(round(X), round(Y), AA_MINE_RADIUS, AA_MINE_RADIUS)
end;

procedure EraseAAMine(X, Y : real);

begin
     setcolor(black);
     setfillstyle(solidfill, black);
     fillellipse(round(X), round(Y), AA_MINE_RADIUS, AA_MINE_RADIUS)
end;

{*******************************************************************}

procedure DrawMMissile(X, Y, theta : real);

begin
     setcolor(Green);
     moveto(round(x), round(y));
     linerel(round(cos(theta) * MEGAMISSILE_LENGTH),
             round(sin(theta) * MEGAMISSILE_LENGTH));
     putpixel(round(x), round(y), LightRed)
end;

procedure EraseMMissile(X, Y, theta : real);

begin
     setcolor(Black);
     moveto(round(x), round(y));
     linerel(round(cos(theta) * MEGAMISSILE_LENGTH),
             round(sin(theta) * MEGAMISSILE_LENGTH))
end;

{*******************************************************************}

procedure DrawHalo(X, Y : real; Color : word);

begin
     setcolor(Color);
     setfillstyle(solidfill, Color);
     fillellipse(round(X), round(Y), HALO_OUTER, HALO_OUTER);
     setcolor(Black);
     setfillstyle(solidfill, black);
     fillellipse(round(X), round(Y), HALO_INNER, HALO_INNER)
end;

procedure EraseHalo(X, Y : real);

begin
     setcolor(black);
     setfillstyle(solidfill, black);
     fillellipse(round(X), round(Y), HALO_OUTER, HALO_OUTER)
end;

{*******************************************************************}

procedure DrawShieldMeter(Amt, Max : integer);

begin
     if Amt < 0 then
        exit;
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     setcolor(LightGray);
     rectangle(5, 5, 105, 15);
     if Amt/Max <= 0.25 then
        setfillstyle(solidfill, lightred)
     else if Amt/Max <= 0.50 then
          setfillstyle(solidfill, yellow)
     else
         setfillstyle(solidfill, lightgreen);
     bar(6, 6, 6 + round(Amt/Max * 100), 14);
     setcolor(LightBlue);
     SetTextStyle(SmallFont, HorizDir, 4);
     SetTextJustify(LeftText, CenterText);
     OutTextXY(110, 10, IntToString(Amt));
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

procedure EraseShieldMeter;

begin
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     setfillstyle(solidfill, black);
     bar(4, 4, 140, 16);
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

{*******************************************************************}

procedure DrawPower(Amt, Max : integer);

begin
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     setcolor(LightGray);
     rectangle(635, 5, 535, 15);
     if Amt/Max <= 0.25 then
        setfillstyle(solidfill, lightred)
     else if Amt/Max <= 0.50 then
          setfillstyle(solidfill, yellow)
     else
         setfillstyle(solidfill, lightgreen);
     bar(634 - round(Amt/Max * 100), 6, 634, 14);
     setcolor(LightBlue);
     SetTextStyle(SmallFont, HorizDir, 4);
     SetTextJustify(RightText, CenterText);
     OutTextXY(530, 10, IntToString(Amt));
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

procedure ErasePower;

begin
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     setfillstyle(solidfill, black);
     bar(636, 4, 500, 16);
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

{*******************************************************************}

procedure EraseCashMeter(Money : longint);

begin
     setfillstyle(SolidFill, Black);
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     bar(295, 4, 340, 16);
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

procedure DrawCashMeter(Money : longint);

begin
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     DrawCash(300, 10);
     settextstyle(SmallFont, HorizDir, 4);
     settextjustify(LeftText, CenterText);
     setcolor(LightGray);
     OutTextXY(300 + CASH_RADIUS + 7, 10, 'X ' + LongToString(Money));
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

{*******************************************************************}

procedure EraseLifeMeter;

begin
     setfillstyle(SolidFill, Black);
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     bar(350, 4, 399, 16);
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

procedure DrawLifeMeter(Lives : integer);

begin
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     setcolor(LightGreen);
     line(355, 6, round(355 - 2.5 * sqrt(3)), 14);
     line(round(355 - 2.5 * sqrt(3)), 14, 355, 11);
     line(355, 11, round(355 + 2.5 * sqrt(3)), 14);
     line(round(355 + 2.5 * sqrt(3)), 14, 355, 6);
     settextstyle(SmallFont, HorizDir, 4);
     settextjustify(LeftText, CenterText);
     setcolor(LightGray);
     OutTextXY(355 + 15, 10, 'X ' + LongToString(Lives));
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

{*******************************************************************}

procedure EraseScoreMeter(Score : longint);

begin
     setfillstyle(SolidFill, Black);
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     bar(400, 4, 480, 16);
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

procedure DrawScoreMeter(Score : Longint);

begin
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     settextstyle(SmallFont, HorizDir, 4);
     settextjustify(RightText, CenterText);
     setcolor(LightGray);
     OutTextXY(480, 10, 'Score: ' + LongToString(Score));
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

{*******************************************************************}

procedure EraseSpecialsMeter(Pilot : Person);

begin
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     setfillstyle(SolidFill, Black);
     bar(169, 4, 294, 16);
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

procedure DrawSpecialsMeter(Pilot : Person);

var
   tstr : string;

begin
     case Pilot.SpePosition of
          0           : tstr := 'NO SPECIALS';
          AAMINE      : tstr := 'AA-MINE X ';
          SUPERSHIELD : tstr := 'SUPER-SHIELD X ';
          MEGAMISSILE : tstr := 'MEGA-MISSILE X '
     end;
     if Pilot.SpePosition <> 0 then
        tstr := tstr + IntToString(Pilot.Specials[Pilot.SpePosition]);
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     settextstyle(SmallFont, HorizDir, 4);
     settextjustify(LeftText, CenterText);
     setcolor(LightCyan);
     OutTextXY(170, 10, tstr);
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

{*******************************************************************}

procedure DrawMeterBox;

begin
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 0, 639, 20, false);
     setcolor(LightMagenta);
     rectangle(0, 0, 639, 20);
     setviewport(0, 0, 639, 479, false);
     setviewport(0, 21, 639, 479, true)
end;

{*******************************************************************}

procedure ShowPilot(Pilot : Person);

var
   tstr : string;

begin
     setcolor(LightGreen);
     settextstyle(SmallFont, HorizDir, 5);
     settextjustify(CenterText, CenterText);
     OutTextXY(320, 120, 'NAME: ' + Pilot.Name);
     case Pilot.Gun of
          1 : tstr := 'CANNON';
          2 : tstr := 'MACHINE GUN';
          3 : tstr := 'POWER CANNON';
          4 : tstr := 'LASER PULSE';
          5 : tstr := '2 X LASER PULSE';
          6, 7, 8, 9 :
              tstr := IntToString(Pilot.Gun - 4) + '-SPRAY'
     end;
     OutTextXY(320, 140, 'WEAPON: ' + tstr);{
                   Specials : SpeType; {Specials posessed
                   SpePosition : integer; {Current special}
     OutTextXY(320, 160, 'SHIELD SYSTEM: LEVEL ' + IntToString(Pilot.ShieldType));
     OutTextXY(320, 180, 'GENERATOR: LEVEL ' + IntToString(Pilot.GeneratorType));
     OutTextXY(320, 200, 'THRUSTERS: LEVEL ' + IntToString(Pilot.Thruster));
     OutTextXY(320, 220, 'LEVEL: ' + LongToString(Pilot.Level));
     if Pilot.Money = 1 then
        OutTextXY(320, 240, 'CASH: 1 CREDIT')
     else
         OutTextXY(320, 240, 'CASH: ' + LongToString(Pilot.Money) + ' CREDITS');
     OutTextXY(320, 260, 'SCORE: ' + LongToString(Pilot.Points));
     OutTextXY(320, 280, 'AA-MINES: ' + IntToString(Pilot.Specials[AAMINE]))
end;

procedure ErasePilot;

begin
     setfillstyle(SolidFill, Black);
     bar(0, 100, 639, 380)
end;

{*******************************************************************}

procedure CheckForNewLife(var Pilot : Person);

begin
     while Pilot.Points >= Pilot.LastLifeUp + LIFE_POINTS do
        begin
             Inc(Pilot.LastLifeUp, LIFE_POINTS);
             Inc(Pilot.Lives);
             EraseLifeMeter;
             DrawLifeMeter(Pilot.Lives)
        end
end;

{*******************************************************************}

procedure FixYou(var You : YShipType; var Pilot : Person);

var
   hr, mn, sc, sc100, curtime : longint;

begin
     setcolor(LightRed);
     setfillstyle(SolidFill, LightRed);
     fillellipse(round(You.Position.X), round(You.Position.Y), 60, 60);
     delay(50);
     setcolor(Black);
     setfillstyle(SolidFill, Black);
     fillellipse(round(You.Position.X), round(You.Position.Y), 60, 60);
     pGetTime(hr, mn, sc, sc100);
     curtime := 360000 * hr + 6000 * mn + 100 * sc + sc100;
     with You do {Initialize your record}
          begin {with}
               Shields := Pilot.MaxShields; {Make shields equal to max. level
                                             of shields}
               Power := Pilot.GeneratorType * 20; {Make power equal to max.
                                                   level of power}
               Bearing.theta := Pi / 2;
               Bearing.mag := 0;
               Position.X := 320;
               Position.Y := 240;
               Aim := Pi/2;
               mov := 0;
               shoot := 0;
               acc := 0;
               turn := 0;
               spec := 0;
               specswitch := 0;
               recharge := 0;
               InvulnerableStay := curtime + DEAD_INVULNERABLE
          end; {with}

     EraseShieldMeter;
     ErasePower;
     EraseCashMeter(Pilot.Money);
     EraseScoreMeter(Pilot.Points);
     EraseSpecialsMeter(Pilot);
     EraseLifeMeter;

     DrawShieldMeter(You.Shields, Pilot.MaxShields); {Draw meters on dashboard}
     DrawPower(You.Power, Pilot.GeneratorType * 20);
     DrawCashMeter(Pilot.Money);
     DrawScoreMeter(Pilot.Points);
     DrawSpecialsMeter(Pilot);
     DrawLifeMeter(Pilot.Lives);
end;

{*******************************************************************}

procedure DrawGun(X, Y, GType : integer);

begin
     setcolor(LightBlue);
     settextstyle(TriplexFont, HorizDir, 3);
     settextjustify(CenterText, CenterText);
     case GType of
          1 : OutTextXY(X, Y, 'CANNON');
          2 : OutTextXY(X, Y, 'MACHINE GUN');
          3 : OutTextXY(X, Y, 'POWER CANNON');
          4 : OutTextXY(X, Y, 'LASER PULSE');
          5 : OutTextXY(X, Y, '2 X LASER PULSE');
          6, 7, 8, 9 :
              OutTextXY(X, Y, IntToString(GType - 4) + '-SPRAY')
     end
end;

procedure EraseGun(X, Y, GType : integer);

begin
     setcolor(Black);
     settextstyle(TriplexFont, HorizDir, 3);
     settextjustify(CenterText, CenterText);
     case GType of
          1 : OutTextXY(X, Y, 'CANNON');
          2 : OutTextXY(X, Y, 'MACHINE GUN');
          3 : OutTextXY(X, Y, 'POWER CANNON');
          4 : OutTextXY(X, Y, 'LASER PULSE');
          5 : OutTextXY(X, Y, '2 X LASER PULSE');
          6, 7, 8, 9 :
              OutTextXY(X, Y, IntToString(GType - 4) + '-SPRAY')
     end
end;

procedure DrawThruster(X, Y, TType : integer);

begin
     setcolor(LightBlue);
     settextstyle(TriplexFont, HorizDir, 3);
     settextjustify(CenterText, CenterText);
     OutTextXY(X, Y, 'LEVEL ' + LongToString(TType))
end;

procedure EraseThruster(X, Y, TType : integer);

begin
     setcolor(Black);
     settextstyle(TriplexFont, HorizDir, 3);
     settextjustify(CenterText, CenterText);
     OutTextXY(X, Y, 'LEVEL ' + LongToString(TType))
end;

procedure DrawShieldDesc(X, Y, SType : integer);

begin
     setcolor(LightBlue);
     settextstyle(TriplexFont, HorizDir, 3);
     settextjustify(CenterText, CenterText);
     OutTextXY(X, Y, 'LEVEL ' + IntToString(SType))
end;

procedure EraseShieldDesc(X, Y, SType : integer);

begin
     setcolor(Black);
     settextstyle(TriplexFont, HorizDir, 3);
     settextjustify(CenterText, CenterText);
     OutTextXY(X, Y, 'LEVEL ' + IntToString(SType))
end;

procedure DrawGenerator(X, Y, GType : integer);

begin
     setcolor(LightBlue);
     settextstyle(TriplexFont, HorizDir, 3);
     settextjustify(CenterText, CenterText);
     OutTextXY(X, Y, 'LEVEL ' + IntToString(GType))
end;

procedure EraseGenerator(X, Y, GType : integer);

begin
     setcolor(Black);
     settextstyle(TriplexFont, HorizDir, 3);
     settextjustify(CenterText, CenterText);
     OutTextXY(X, Y, 'LEVEL ' + IntToString(GType))
end;

procedure DrawSpec(X, Y, SType : integer);

begin
     setcolor(LightCyan);
     settextstyle(TriplexFont, HorizDir, 3);
     settextjustify(CenterText, CenterText);
     case SType of
          1 : OutTextXY(X, Y, 'ANTI-ALIEN MINE');
          2 : OutTextXY(X, Y, 'SUPER SHIELD');
          3 : OutTextXY(X, Y, 'MEGA MISSILE')
     end
end;

procedure EraseSpec(X, Y, SType : integer);

begin
     setcolor(Black);
     settextstyle(TriplexFont, HorizDir, 3);
     settextjustify(CenterText, CenterText);
     case SType of
          1 : OutTextXY(X, Y, 'ANTI-ALIEN MINE');
          2 : OutTextXY(X, Y, 'SUPER SHIELD');
          3 : OutTextXY(X, Y, 'MEGA MISSILE')
     end
end;

{*******************************************************************}

procedure CheckShieldSuckerHit(var Projs : ProjPtr; var hitbullet : boolean; var cur : EShipPtr; var Pilot : Person;
                               var Sound : Sounds; NoSFX : boolean);

var
   bul, del : ProjPtr;
   del2 : EShipPtr;

begin
     bul := Projs^.next;
     while (bul <> Projs) and not(hitbullet) do
           begin
                case bul^.what of
                     BULLET :
                           if sqrt(sqr(cur^.Position.X - bul^.Position.X) +
                             sqr(cur^.Position.Y - bul^.Position.Y)) <= 3 + SHIELD_SUCKER_RADIUS then
                              begin
                                   EraseBullet(bul^.Position.X, bul^.Position.Y);
                                   cur^.Shields := cur^.Shields - 1;
                                   del := bul;
                                   bul := bul^.prev;
                                   bul^.next := del^.next;
                                   bul^.next^.prev := bul;
                                   dispose(del);
                                   if not(NoSFX) and SoundPlaying(HIT_NOISE) then
                                      StopSound(HIT_NOISE);
                                   if not(NoSFX) then
                                      StartSound(Sound[HIT_NOISE], HIT_NOISE, false)
                              end;
                     POWERGUN_BULLET :
                           if sqrt(sqr(cur^.Position.X - bul^.Position.X) +
                             sqr(cur^.Position.Y - bul^.Position.Y)) <= 3 + SHIELD_SUCKER_RADIUS then
                              begin
                                   EraseBullet(bul^.Position.X, bul^.Position.Y);
                                   cur^.Shields := cur^.Shields - 5;
                                   del := bul;
                                   bul := bul^.prev;
                                   bul^.next := del^.next;
                                   bul^.next^.prev := bul;
                                   dispose(del);
                                   if not(NoSFX) and SoundPlaying(HIT_NOISE) then
                                      StopSound(HIT_NOISE);
                                   if not(NoSFX) then
                                      StartSound(Sound[HIT_NOISE], HIT_NOISE, false)
                              end;
                     LASERPULSE_PULSE :
                           if sqrt(sqr(cur^.Position.X - bul^.Position.X) +
                             sqr(cur^.Position.Y - bul^.Position.Y)) <= 3 + SHIELD_SUCKER_RADIUS then
                              begin
                                   EraseLaserpulsePulse(bul^.Position.X, bul^.Position.Y, bul^.Bearing);
                                   cur^.Shields := cur^.Shields - 2;
                                   del := bul;
                                   bul := bul^.prev;
                                   bul^.next := del^.next;
                                   bul^.next^.prev := bul;
                                   dispose(del);
                                   if not(NoSFX) and SoundPlaying(HIT_NOISE) then
                                      StopSound(HIT_NOISE);
                                   if not(NoSFX) then
                                      StartSound(Sound[HIT_NOISE], HIT_NOISE, false)
                              end;
                     SPE_OFFSET + MEGAMISSILE :
                              if (abs(cur^.Position.X - bul^.Position.X) < SHIELD_SUCKER_RADIUS +
                                 MEGAMISSILE_LENGTH) and
                                 (abs(cur^.Position.Y - bul^.Position.Y) < SHIELD_SUCKER_RADIUS +
                                 MEGAMISSILE_LENGTH) then {If it hit}
                                   begin {if}
                                        EraseMMissile(bul^.Position.X, bul^.Position.Y, bul^.Bearing.theta);
                                          {erase mega missile}
                                        MMExplode(bul);
                                        cur^.Shields := 0
                                   end {If}
                end;
           if cur^.Shields <= 0 then
              begin
                   if not(NoSFX) and SoundPlaying(EXP_SND) then
                      StopSound(EXP_SND);
                   if not(NoSFX) then
                      StartSound(Sound[EXP_SND], EXP_SND, false);
                   EraseScoreMeter(Pilot.Points); {Update score meter}
                   Pilot.Points := Pilot.Points + SHIELD_SUCKER_POINT_VALUE;
                    {Update points}
                   CheckForNewLife(Pilot); {Check to see if you got a new life}
                   DrawScoreMeter(Pilot.Points); {Update score meter}
                   setcolor(LightRed);
                   setfillstyle(SolidFill, LightRed);
                   fillellipse(round(cur^.Position.X), round(cur^.Position.Y), 60, 60);
                   delay(50);
                   setcolor(Black);
                   setfillstyle(SolidFill, Black);
                   fillellipse(round(cur^.Position.X), round(cur^.Position.Y), 60, 60);
                   del2 := cur;
                   cur := cur^.prev;
                   cur^.next := del2^.next;
                   cur^.next^.prev := cur;
                   dispose(del2);
                   hitbullet := true
              end;
           bul := bul^.next
     end
end;

{*******************************************************************}

procedure DrawShieldSucker(X, Y : real);

begin
     setfillstyle(SolidFill, Red);
     setcolor(Yellow);
     fillellipse(round(X), round(Y), SHIELD_SUCKER_RADIUS, SHIELD_SUCKER_RADIUS)
end;

procedure EraseShieldSucker(X, Y : real);

begin
     setfillstyle(SolidFill, Black);
     setcolor(Black);
     fillellipse(round(X), round(Y), SHIELD_SUCKER_RADIUS, SHIELD_SUCKER_RADIUS)
end;

procedure MSplit(var Missile : ProjPtr);

var
   ins : ProjPtr;
   ctr : integer;

begin
     for ctr := 1 to MSPLIT_NUM - 1 do
         begin
              new(ins);
              with ins^ do
                   begin
                        what := SPE_OFFSET + MEGAMISSILE;
                        range := Missile^.range;
                        Bearing.mag := Missile^.Bearing.mag;
                        Bearing.theta := Missile^.Bearing.theta - (MSPLIT_NUM - 1) * MSPLIT_ANGLE/2 +
                                         (ctr - 1) * MSPLIT_ANGLE;
                        Position := Missile^.Position;
                        move := 0;
                        prev := Missile;
                        next := Missile^.next
                   end;
              Missile^.next := ins;
              ins^.next^.prev := ins
         end;
     Missile^.Bearing.theta := Missile^.Bearing.theta - (MSPLIT_NUM - 1) * MSPLIT_ANGLE/2 +
                                         (3) * MSPLIT_ANGLE
end;

procedure MMExplode(var Missile : ProjPtr);

var
   ins : ProjPtr;
   ctr : integer;

begin
     for ctr := 1 to NumExplode - 1 do
         begin
              new(ins);
              with ins^ do
                   begin
                        what := SPE_OFFSET + MEGAMISSILE;
                        range := MMEXPLODE_RANGE;
                        Bearing.mag := Missile^.Bearing.mag;
                        Bearing.theta := ctr * Pi/2;
                        Position := Missile^.Position;
                        move := 0;
                        prev := Missile;
                        next := Missile^.next
                   end;
              Missile^.next := ins;
              ins^.next^.prev := ins
         end;
     Missile^.Bearing.theta := 0;
     Missile^.range := MMEXPLODE_RANGE
end;

procedure DrawStar(X, Y : real);

var
   col : word;

begin
     case random(5) of
          0 : col := LightCyan;
          1 : col := LightBlue;
          2 : col := LightGreen;
          3 : col := LightRed;
          4 : col := White
     end;
     putpixel(round(X), round(Y), col)
end;

procedure EraseStar(X, Y : real);

begin
     putpixel(round(X), round(Y), Black)
end;

procedure ShowCredits(var EscFlag : boolean);

{Pre:  None.
 Post: Shows credits on screen.}

begin {ShowCredits}
     cleardevice;
     setcolor(LightGreen);
     settextstyle(TriplexFont, HorizDir, 5); {Set up fonts}
     settextjustify(CenterText, CenterText);
     OutTextXY(320, 50, 'Alien Attack Credits'); {Show credits}
     line(50, 80, 590, 80); {Underline}
     setcolor(LightCyan);
     settextstyle(TriplexFont, HorizDir, 3);
     OutTextXY(320, 100, 'Programmer: Po-Shen Loh');
     OutTextXY(320, 140, 'Designed game engine.');
     line(50, 170, 590, 170); {Underline}
     setcolor(LightMagenta);
     OutTextXY(320, 200, 'Help from: Ethan Brodsky');
     OutTextXY(320, 240, 'Enabled me to add sound to Alien Attack.');
     OutTextXY(320, 270, 'Also helped enhance keyboard support.');
     line(50, 300, 590, 300); {Underline}
     setcolor(Yellow);
     OutTextXY(320, 330, 'Idea from: Shawn Jeffery');
     OutTextXY(320, 370, 'Supplied initial idea.');
     settextstyle(SmallFont, HorizDir, 4);
     settextjustify(LeftText, CenterText);
     setcolor(LightRed);
     OutTextXY(20, 470, 'Press ESCAPE to return to main menu.'); {Prompt}
     repeat {Wait until Escape is pressed}
     until EscFlag
end; {ShowCredits}

procedure SetUp(var par : paratype);

var
   tstr : array[1..7] of string;

begin
     par[1] := 'By Po-Shen Loh                                              '
end;

end.