program Alien_Attack;

{********************************************************************
NAME: Po-Shen Loh
CLASS: Pascal
PROJECT NAME: Alien Attack!
PROGRAM DESCRIPTION: Asteroids clone.
GLOBAL FLAGS USED: Byte flag stored absolute $40:$17, sound flags,
       and termination flags used for multiple module terminations.
SPECIAL IDENTIFIERS USED: Halt and Exit, both to ensure proper
        execution.
INPUT: Game play.
OUTPUT: Fun.
********************************************************************}

uses CRT, {Links in screen unit.  Enables calls for screen clearing and color
           changing}
     DOS, {Links in DOS unit.  Used for time functions}
     graph, {Links in BGI graphics pack.  Used for graphics}
     PFontV13, {Links in Po-Shen's font library and animation kit.  Also
                enables graphical reading.}
     PButV103, {Links in Po-Shen's old button library.}
     PButV104, {Links in Po-Shen's button library.  It allows easy buttons
                by implementing them inside objects, in the private sector.}
     PMseV101, {Links in Po-Shen's mouse library.  It enables mouse calls.}
     drivers, {Links in the drivers unit.  Allows trapping of CTRL, ALT,
               SHIFT, etc.}
     PSndV102, {Links in Po-Shen's sound blaster collection, created
               mainly by Ethan Brodsky.  Sound file copy protection
               provided by Po-Shen Loh.}
     AliDraw; {Links in the alien drawing routines}

var
   BaseIO : word; {Base IO address of sound}
   IRQ, DMA, DMA16 : byte; {IRQ, DMA, DMA16 of sound}
   Sound : Sounds; {The sounds to be played}
   OldExitProc : pointer; {Pointer to linked list of exit procedures}
   NoMusic, NoSFX : boolean; {Global sound flags}
   MemGuard : pointer; {Guards heap against crashes.}
   AutoDetectPointer : pointer; {Graph. det. ptr.}

   Fnt : FontObject; {Used to link in graphics}

   Pilot : Person; {Record of current pilot}

   SoundsUsed : boolean; {Flag for sounds}

   CtrlFlag, UpFlag, DownFlag, LeftFlag, RightFlag, EscFlag,
   LShiftFlag, RShiftFlag, AltFlag, TabFlag : boolean; {Shift flags}

   OldKbdVec : pointer; {Old keyboard vector}

   NormalOperation : boolean; {Stores whether the keyboard will return data
                               normally}

   Shifted : byte absolute $40:$17; {Keyboard shift byte flag}

   GrafixOn : boolean; {Graphics flag}

   TESTING : boolean; {FIX}

{$F+} {Far calls on}

procedure CallOldInt(Sub : pointer);

{Desc: Calls old interrupt vector pointed to by Sub
 Pre : Sub points to interrupt vector.
 Post: Interrupt vector called}

begin {CallOldInt}
     inline($9C/          { PUSHF (Assembler) }
            $FF/$5E/$06)  { CALL DWORD PTR [BP+6] (Assembler) }
end; {CallOldInt}

procedure Keyboard(Flags, CS, IP, AX, BX, CX, DX, SI, DI, DS, ES, BP : word); interrupt;

{Desc: My keyboard event handler
 Pre : All flags store correct values and key pressed.
 Post: All flags store new values and key press removed if normal operation}

begin {Keyboard}
     if hi(AX) = $4F then {If this interrupt is from the keyboard}
        case lo(ax) of {Branch based on what key was pressed}
             _UP           : UpFlag     := true;
             _DOWN         : DownFlag   := true;
             _LEFT         : LeftFlag   := true;
             _RIGHT        : RightFlag  := true;
             _CTRL         : CtrlFlag   := true;
             _ESC          : EscFlag    := true;
             _ALT          : AltFlag    := true;
             _LSHIFT       : LShiftFlag := true;
             _RSHIFT       : RShiftFlag := true;
             _TAB          : TabFlag    := true;
             _UP + 128     : UpFlag     := false;
             _DOWN + 128   : DownFlag   := false;
             _LEFT + 128   : LeftFlag   := false;
             _RIGHT + 128  : RightFlag  := false;
             _CTRL + 128   : CtrlFlag   := false;
             _ALT + 128    : AltFlag    := false;
             _ESC + 128    : EscFlag    := false;
             _LSHIFT + 128 : LShiftFlag := false;
             _RSHIFT + 128 : RShiftFlag := false;
             _TAB + 128    : TabFlag    := false
             else {Do nothing}
        end; {case}
     CallOldInt(OldKbdVec); {Call old vector}
     if keypressed and not(NormalOperation) then
        while keypressed do {Remove key presses from buffer if normal operation}
              readkey
end; {Keyboard}

{$F-} {Far calls off}

function GetPossPath(SArr : array of string; fname : string) : string;

{Desc: Finds a possible path for the specified file
 Pre : None.
 Post: Returns possible path for specified file}

var
   ctr : integer; {Loop counter}
   tfile : file; {Temp. file}

begin {GetPossPath}
     for ctr := 0 to high(SArr) do {Searches through array}
         begin {for}
              assign(tfile, SArr[ctr]+fname);
              {$I-} {IO checking off}
              reset(tfile, 1);
              {$I+} {IO checking on}
              if IOResult = 0 then {If no error}
                 begin {if}
                      close(tfile);
                      GetPossPath := SArr[ctr]+fname; {Return path}
                      exit {Leave proc.}
                 end {if}
         end; {for}
     GetPossPath := fname {Return unmodified file name}
end; {GetPossPath}

function LookForSounds : boolean;

{Desc: Searches for sound files and command line flags
 Pre : None.
 Post: Returns whether sounds are present.}

var
   test : file; {temp. file}
   None : boolean; {Flag}

begin {LookForSounds}
     None := false;
     if ParamCount > 0 then  {Counts params.}
        if ParamStr(1) = '-s' then {Tests params.}
           None := true
        else if ParamStr(1) = '-S' then
             None := true;
     {$I-} {IO OFF}
     assign(test, GetPossPath(SoundPaths, 'MUSIC.DAT'));
     reset(test, 1);
     if IOResult <> 0 then
        NoMusic := true {Modify global flag}
     else
         close(test);
     assign(test, GetPossPath(SoundPaths, 'SFX.DAT'));
     reset(test, 1);
     if IOResult <> 0 then
        NoSFX := true {Modify global flag}
     else
         close(test);
     {$I+} {IO on}
     LookForSounds := not(None or NoMusic and NoSFX) {Return}
end; {LookForSounds}

function UpCaseStr(s : string) : string;

{Desc: Converts any string to all caps.
 Pre : S is a string.
 Post: Returns string in all caps.}

var
   ctr : integer; {Counter}

begin {UpCaseStr}
     for ctr := 1 to ord(s[0]) do
         s[ctr] := upcase(s[ctr]); {Converts char. to upcase}
     UpCaseStr := s {Returns converted string}
end; {UpCaseStr}

function HexW(W : word) : string; {Word}

{====================================================================
FUNCTION DESCRIPTION: Converts a number from base 10 to base 16.
INPUT: A positive base 10 integer from calling module.
OUTPUT: A string containing the hexadecimal equivalent of the passed
        number to calling module.
====================================================================}

const
     HexChars : array [0..$F] of Char = '0123456789ABCDEF'; {Array of valid
                                                             hexadecimal
                                                             characters.}

begin {HexW}
     HexW := HexChars[(W and $F000) shr 12] + { Converts the number from}
             HexChars[(W and $0F00) shr 8]  + { base 10 to base 16.}
             HexChars[(W and $00F0) shr 4]  +
             HexChars[(W and $000F)]
end; {HexW}

procedure ClearProc; far;

{Auto terminate procedure.  In far call mode.}

begin {ClearProc}
     if GrafixOn then
        closegraph;
     release(MemGuard); {Restores memory condition}
     textcolor(LightGray); {Set to original colors}
     textbackground(Black);
     writeln;
     writeln('Goodbye!');
     writeln;
     SetIntVec(KbdInt, OldKbdVec);
     ExitProc := OldExitProc {Chain to next exit procedure}
end; {ClearProc}

procedure OurExitProc; far;

{====================================================================
PROCEDURE DESCRIPTION: If the program terminates with a runtime error
          before the extended memory is deallocated, then the memory
          will still be allocated, and will be lost until the next
          reboot.  This exit procedure is ALWAYS called upon program
          termination and will de-allocate extended memory if
          necessary.
INPUT: Pointer to the new exit procedure from calling module.
OUTPUT: None.
====================================================================}

var
   ctr : integer; {Loop counter}

begin {OurExitProc}
     for ctr := 1 to NumSounds do
         if Sound[ctr] <> nil then {If the memory allocated for the sound to
                                    be played has not yet been de-allocated...}
            FreeSound(Sound[ctr]); {De-allocates memory for the sound}
     if SharedEMB then {If the sounds are being stored in a shared EMB...}
        ShutdownSharing; {The sharing is stopped.}
     ExitProc := @ClearProc;
end; {OurExitProc}

procedure InitSound(var BaseIO : word; IRQ, DMA, DMA16 : byte; var Sound : Sounds; SoundPaths : array of string);

{====================================================================
PROCEDURE DESCRIPTION: Initializes the sound unit.
INPUT: BaseIO, IRQ, DMA, DMA16, sound variables from calling module.
OUTPUT: Modified BaseIO, IRQ, DMA, and DMA16 vars. to calling module.
====================================================================}

begin {InitSound}
     writeln; {Outputs a blank line}
     writeln('-------------------------------------------');
     writeln('Sound Mixing Library v1.27 by Ethan Brodsky'); {Credits}
     if not(GetSettings(BaseIO, IRQ, DMA, DMA16)) then {If the sound was not
                                                        detected...}
        begin {if}
             writeln('Error initializing:  Invalid or non-existant BLASTER environment variable');
                     {error message}
             writeln('Press ENTER.');
             readln;
             NoSFX := true;
             NoMusic := true;
             exit {BLASTER environment variable invalid or non-existant}
        end {if}
     else
         begin {else}
              if not(InitSB(BaseIO, IRQ, DMA, DMA16)) then {If the BLASTER
                                                            environment var.
                                                            was set
                                                            incorrectly...}
                 begin {if}
                      writeln('Error initializing sound card');{Error message}
                      writeln('Incorrect base IO address, sound card not installed, or broken');
                      writeln('Press ENTER.');
                      readln;
                      NoSFX := true;
                      NoMusic := true;
                      exit {Sound card could not be initialized}
                 end; {if}
              if SixteenBit then {If the sound card supports 16-bit mode...}
                 writeln('BaseIO=', HexW(BaseIO), 'h    IRQ', IRQ, '    DMA8=', DMA, '    DMA16=', DMA16)
                    {Stats. of 16-bit sound card.}
              else {8-bit}
                  writeln('BaseIO=', HexW(BaseIO), 'h        IRQ', IRQ, '        DMA8=', DMA)
                     {Stats. of 8-bit sound card.}
         end; {else}
     write('DSP version ', DSPVersion:0:2, ':  '); {Outputs the DSP version}
     if SixteenBit then {If it was 16-bit...}
        write('16-bit, ') {Notification}
     else {8-bit...}
         write('8-bit, '); {Notification}
     if AutoInit then {If it was auto-initialized...}
        writeln('Auto-initialized') {Notification}
     else {It is single-cycle...}
         writeln('Single-cycle'); {Notification}
     if not(InitXMS) then {This loads the sounds into extended memory.  If
                           there isn't any...}
        begin {if}
             writeln('Error initializing extended memory'); {Error message}
             writeln('HIMEM.SYS must be installed');
             writeln('Press ENTER.');
             NoSFX := true;
             NoMusic := true;
             exit {XMS driver not installed}
        end {if}
     else
         begin {else}
              writeln('Extended memory succesfully initialized');
              write('Free XMS memory:  ', GetFreeXMS, 'k  '); {Notification}
              if GetFreeXMS < XMSRequired then {Not enough XMS...}
                 begin {if}
                      writeln('Insufficient free XMS'); {Error message}
                      writeln('You are might be running this program from the protected mode IDE.');
                      writeln('Run it from the command line.');
                      writeln('If that doesn''t work, then try using a different machine');
                      writeln('that has more free XMS.');
                      writeln('The sounds will not be loaded.');
                      writeln('Press ENTER.');
                      readln;
                      NoSFX := true;
                      NoMusic := true;
                      exit {Insufficient XMS memory}
                 end {if}
              else
                  begin {else}
                       if SharedEMB then{If the sounds are to share an EMB...}
                          InitSharing; {Initialize the sharing}

                       writeln('Loading sounds...');

                       if not(NoMusic) then begin

                          OpenSoundResourceFile(GetPossPath(SoundPaths, 'MUSIC.DAT'), 3); {Opens the sound
                                                            file.}

                          LoadSound(Sound[1], 'Background'); {Loads music}

                          CloseSoundResourceFile {Closes the file}

                       end;

                       if not(NoSFX) then begin

                          OpenSoundResourceFile(GetPossPath(SoundPaths, 'SFX.DAT'), 3); {Opens the sound
                                                             file.}

                          LoadSound(Sound[2], 'Machine_Gun'); {Loads SFX}
                          LoadSound(Sound[3], 'Explosion');
                          LoadSound(Sound[4], 'FusionGun');
                          LoadSound(Sound[5], 'PhotonGun');
                          LoadSound(Sound[6], 'Hit_1');
                          LoadSound(Sound[7], 'Exorcism');
                          LoadSound(Sound[8], 'Vision');

                          CloseSoundResourceFile {Closes the file}

                       end;

                       OldExitProc := ExitProc; {Adds the exit procedure to
                                                 the list of others.}
                       ExitProc := @OurExitProc {Assigns the pointer to the
                                                 address of the exit procedure}
                  end {else}
         end; {else}
     InitMixing {Allocates internal buffers and starts digitized sound
                  output}
end; {InitSound}

procedure ShutdownSound(var Sound : Sounds);

{====================================================================
PROCEDURE DESCRIPTION: De-allocates memory for sound, shuts down EMB
                       sharing and frees all allocated extended
                       memory, if EMB sharing is on.
INPUT: The sound pointer from calling module.
OUTPUT: None.
====================================================================}

var
   ctr : integer;

begin {ShutDownSound}
     ShutdownMixing; {Deallocates internal buffers and stops digitized sound
                      output}
     ShutdownSB; {Removes interrupt handler and resets DSP}

     for ctr := 1 to NumSounds do {De-allocates all memory for sounds}
         if Sound[ctr] <> nil then {If the memory for the sound hasn't been
                                    de-allocated yet...}
            FreeSound(Sound[ctr]); {De-allocates memory for sound.}
     if SharedEMB then {If the sounds are in a shared EMB...}
        ShutdownSharing {Shuts down the EMB sharing.}
end; {ShutDownSound}

procedure AddVecs(theta1, mag1, theta2, mag2 : real; var V : Vector);

{Pre : None.
 Post: Vectors are added and returned in V}

var
   X, Y : real; {Coordinates}

begin {AddVecs}
     X := mag1 * cos(theta1) + mag2 * cos(theta2);
     Y := mag1 * sin(theta1) + mag2 * sin(theta2);
     V.mag := sqrt(sqr(X) + sqr(Y));
     if X > 0 then
        V.theta := arctan(Y/X)
     else if (X = 0) and (Y >= 0) then
          V.theta := Pi/2
     else if (X = 0) and (Y < 0) then
          V.theta := 3 * Pi/2
     else
         V.theta := arctan(Y/X) + Pi; {Pi is 3.14}
     if V.theta < 0 then
        V.theta := V.theta + 2 * Pi
end; {AddVecs}

function Dist(X1, Y1, X2, Y2 : real) : real;

{Pre : None.
 Post: Returns distance between (X1, Y1) AND (X2, Y2) with math}

begin {Dist}
     Dist := sqrt(sqr(X2 - X1) + sqr(Y2 - Y1))
end; {Dist}

{$I ALIGUNS.PAS} {Links in guns file}
{$I ALINMES.PAS} {Links in enemies file}
{$I ALISHOP.PAS} {Links in shop file}

procedure Init(var Asts : AstPtr; var Goods : GoodPtr; var Enemies : EShipPtr;
               var You : YShipType; var Projs : ProjPtr; var Pilot : Person; var MaxLevel : integer;
               var Lev : LevelType);

{Pre : None.
 Post: All arguments are initialized.}

var
   ctr : longint; {Loop counter}
   Rock : AstPtr; {Temp. asteroid addition ptr.}
   IniFile : text; {Files}
   LevFile : text;
   LevFileStr : string; {Name of level file}
   hr, mn, sc, sc100, curtime : LONGINT; {FIX}

begin {Init}
     new(Asts);
     Asts^.Size := -1;
     Asts^.prev := Asts;
     Asts^.next := Asts;
     Rock := Asts;

     assign(IniFile, GetPossPath(IniPaths, IniFileName)); {Gets a path for the
                                                           .INI file}
     reset(IniFile);
     readln(IniFile, MaxLevel);
     for ctr := 1 to Pilot.level do {Get the name of the level file}
         readln(IniFile, LevFileStr);

     close(IniFile);

     assign(LevFile, LevFileStr); {Link level file to name}
     reset(LevFile);
     readln(LevFile, Lev.NumAsts);     {get data}
     readln(LevFile, Lev.AlienChance);
     readln(LevFile, Lev.AlienLevel);
     close(LevFile);

     for ctr := 1 to Lev.NumAsts do {Add asteroids}
         begin {for}
              new(Rock^.next);
              with Rock^.next^ do {Assign values to asteroid}
                   begin {with}
                        Size := LARGE;
                        Bearing.theta := random(360) * Pi / 180;
                        Bearing.mag := random(3) + 1;
                        repeat {Get coordinates that are not on your ship}
                              Position.X := random(640);
                              Position.Y := random(MaxGameY)
                        until not((abs(Position.X - 320) < 50) and
                                  (abs(Position.Y - 240) < 50));
                        Move := 0;
                        Prev := Rock
                   end; {with}
              rock := rock^.next;
              Asts^.prev := rock
         end; {for}
     rock^.next := Asts;

     new(Goods);
     Goods^.what := HEADER;
     Goods^.prev := Goods;
     Goods^.next := Goods;

     new(Enemies);
     Enemies^.prev := Enemies;
     Enemies^.next := Enemies;

     with You do {Initialize your record}
          begin {with}
               Shields := Pilot.MaxShields; {Make shields equal to max. level
                                             of shields}
               Power := Pilot.GeneratorType * 20; {Make power equal to max.
                                                   level of power}
               Bearing.theta := Pi / 2;
               Bearing.mag := 0;
               Position.X := 320;
               Position.Y := 240;
               Aim := Pi/2;
               mov := 0;
               shoot := 0;
               acc := 0;
               turn := 0;
               spec := 0;
               specswitch := 0;
               recharge := 0;
               if TESTING then
                  begin
                       pGetTime(hr, mn, sc, sc100); {FIX}
                       curtime := 360000 * hr + 6000 * mn + 100 * sc + sc100;
                       InvulnerableStay := curtime + 6000 {FIX}
                  end
               else
                   invulnerablestay := 0
          end; {with}

     DrawShieldMeter(You.Shields, Pilot.MaxShields); {Draw meters on dashboard}
     DrawPower(You.Power, Pilot.GeneratorType * 20);
     DrawCashMeter(Pilot.Money);
     DrawScoreMeter(Pilot.Points);
     DrawSpecialsMeter(Pilot);
     DrawLifeMeter(Pilot.Lives);

     new(Projs);
     Projs^.what := HEADER;
     Projs^.prev := Projs;
     Projs^.next := Projs
end; {Init}

procedure AnimateY(var You : YShipType; var Pilot : Person; var bak2shop, dead : boolean);

{Pre : "You" is defined. "Pilot" is initialized.
 Post: If your ship needs to move, then it moves.  Also regenerates power bar.}

var
   hr, mn, sc, sc100 : longint; {Time}
   curtime : longint; {Time counter}

begin {AnimateY}
     pGetTime(hr, mn, sc, sc100); {Get the time}
     curtime := 360000 * hr + 6000 * mn + 100 * sc + sc100; {Init the time counter}
     with You do {Animate your ship}
          begin {with}
               if mov + Y_Move_Delay <= curtime then {If time to move}
                  begin {if}
                       mov := curtime;
                       if curtime <= InvulnerableStay then {If you are invulnerable}
                          EraseIYShip(Position.X, Position.Y, Aim) {Erase invulnerable ship}
                       else
                           EraseYShip(Position.X, Position.Y, Aim); {Erase your ship}
                       Position.X := Position.X + cos(Bearing.theta) * Bearing.mag * Y_MOVE_COEFF;
                       Position.Y := Position.Y + sin(Bearing.theta) * Bearing.mag * Y_MOVE_COEFF;
                       if Position.X > 640 then                 {If you go off the screen, then your coords are modified.}
                          Position.X := Position.X - 640
                       else if Position.X <= 0 then
                            Position.X := Position.X + 640;
                       if Position.Y > MaxGameY then
                          Position.Y := Position.Y - MaxGameY
                       else if Position.Y <= 0 then
                            Position.Y := Position.Y + MaxGameY;
                       if curtime <= InvulnerableStay then {If you are invulnerable}
                          if curtime + WARNING_TIME >= InvulnerableStay then {If 2 secs. to off invulnerability}
                             DrawWYShip(Position.X, Position.Y, Aim) {Draw your invulnerable ship in warning mode}
                          else
                              DrawIYShip(Position.X, Position.Y, Aim) {Draw your invulnerable ship} {Hello, Mr. Williams!}
                       else
                           DrawYShip(Position.X, Position.Y, Aim) {Draw your ship}
                  end; {with}
               if (recharge + Pilot.RechargeDelay <= curtime) and (You.Power < Pilot.GeneratorType * 20) then
                {If we need to recharge your power}
                  begin {if}
                       recharge := curtime;
                       ErasePower; {Update power meter}
                       You.Power := You.Power + RECHARGE_AMT;
                       if You.Power > Pilot.GeneratorType * 20 then
                          You.Power := Pilot.GeneratorType * 20;
                       DrawPower(You.Power, Pilot.GeneratorType * 20) {Update power meter}
                  end {if}
          end {with}
end; {AnimateY}

procedure AnimateP(var Projs : ProjPtr; var You : YShipType; var EShips : EShipPtr; var bak2shop, dead : boolean);

{Pre : "Pilot" is defined.  "You" is defined.  "EShips" is defined.
 Post: All projectiles are moved.  Crashing into you is trapped.}

var
   cur : ProjPtr; {traversion ptr.}
   del : ProjPtr; {Deletion ptr.}
   curtime, hr, mn, sc, sc100 : longint; {Time vars.}
   del2 : EShipPtr; {Deletion ptr.}
   theta : real; {Angle}

begin {AnimateP}
     pGetTime(hr, mn, sc, sc100); {Get time}
     curtime := 360000 * hr + 6000 * mn + 100 * sc + sc100; {Modify time ctr.}
     cur := Projs^.next;
     while cur <> Projs do {animate ALL projectiles}
           begin {while}
                case cur^.what of {Branch based on what projectile is}
                     BULLET : begin {if it is a bullet}
                                   if cur^.move + BULLET_TRAVEL_DELAY <= curtime then {if time to animate}
                                      begin {if}
                                           cur^.move := curtime;
                                           EraseBullet(cur^.Position.X, cur^.Position.Y); {Erase bullet}
                                           cur^.Position.X := cur^.Position.X + cos(cur^.Bearing.theta) * cur^.Bearing.mag;
                                           cur^.Position.Y := cur^.Position.Y + sin(cur^.Bearing.theta) * cur^.Bearing.mag;
                                           if cur^.Position.X > 640 then                 {Modify if off screen}
                                              cur^.Position.X := cur^.Position.X - 640
                                           else if cur^.Position.X <= 0 then
                                                cur^.Position.X := cur^.Position.X + 640;
                                           if cur^.Position.Y > MaxGameY then
                                              cur^.Position.Y := cur^.Position.Y - MaxGameY
                                           else if cur^.Position.Y <= 0 then
                                                cur^.Position.Y := cur^.Position.Y + MaxGameY;
                                           cur^.range := cur^.range - 1;
                                           if (cur^.range >= 0) then {If we need to draw bullet}
                                              DrawBullet(cur^.Position.X, cur^.Position.Y) {Draw bullet}
                                           else {Delete bullet}
                                               begin {else}
                                                    del := cur;
                                                    cur := cur^.prev;
                                                    cur^.next := del^.next;
                                                    cur^.next^.prev := cur;
                                                    dispose(del)
                                               end {else}
                                      end; {if}
                                   if (abs(cur^.Position.X - You.Position.X) < 12) and
                                    (abs(cur^.Position.Y - You.Position.Y) < 12) then {If bullet crashed into you}
                                      begin {if}
                                           if SoundPlaying(HIT_NOISE) then {If the hit noise is playing then stop it}
                                              StopSound(HIT_NOISE);
                                           StartSound(Sound[HIT_NOISE], HIT_NOISE, false); {Start hit noise}
                                           if not(You.InvulnerableStay >= curtime) then
                                              You.Shields := You.Shields - 1; {Decrement shields}
                                           EraseShieldMeter; {Modify dashboard}
                                           DrawShieldMeter(You.Shields, Pilot.MaxShields); {Modify dashboard}
                                           if You.Shields <= 0 then {If you are out of shields}
                                              if Pilot.Lives = 0 then
                                                 dead := true {You are dead}
                                              else
                                                  begin
                                                       EraseYShip(You.Position.X, You.Position.Y, You.Aim);
                                                       Dec(Pilot.Lives);
                                                       FixYou(You, Pilot)
                                                  end; {else}
                                           EraseBullet(cur^.Position.X, cur^.Position.Y); {Erase bullet that killed you}
                                           del := cur; {Remove bullet from linked list}
                                           cur := cur^.prev;
                                           cur^.next := del^.next;
                                           cur^.next^.prev := cur;
                                           dispose(del)
                                      end {if}
                              end; {BULLET}
                     POWERGUN_BULLET :
                              begin {if it is a powergun bullet}
                                   if cur^.move + BULLET_TRAVEL_DELAY <= curtime then {If time to move}
                                      begin {if}
                                           cur^.move := curtime;
                                           ErasePBullet(cur^.Position.X, cur^.Position.Y); {Erase power bullet}
                                           cur^.Position.X := cur^.Position.X + cos(cur^.Bearing.theta) * cur^.Bearing.mag;
                                           cur^.Position.Y := cur^.Position.Y + sin(cur^.Bearing.theta) * cur^.Bearing.mag;
                                           if cur^.Position.X > 640 then                  {If off screen then modify}
                                              cur^.Position.X := cur^.Position.X - 640
                                           else if cur^.Position.X <= 0 then
                                                cur^.Position.X := cur^.Position.X + 640;
                                           if cur^.Position.Y > MaxGameY then
                                              cur^.Position.Y := cur^.Position.Y - MaxGameY
                                           else if cur^.Position.Y <= 0 then
                                                cur^.Position.Y := cur^.Position.Y + MaxGameY;
                                           cur^.range := cur^.range - 1;
                                           if (cur^.range >= 0) then {If we must draw power bullet}
                                              DrawPBullet(cur^.Position.X, cur^.Position.Y) {Draw power bullet}
                                           else
                                               begin {else}
                                                    del := cur; {Delete it}
                                                    cur := cur^.prev;
                                                    cur^.next := del^.next;
                                                    cur^.next^.prev := cur;
                                                    dispose(del)
                                               end {else}
                                      end {if}
                              end; {POWERGUN_BULLET}
                     LASERPULSE_PULSE :
                              begin {if it's a laser pulse}
                                   if cur^.move + LASERPULSE_TRAVEL_DELAY <= curtime then {If it moves}
                                      begin {if}
                                           cur^.move := curtime;
                                           EraseLaserpulsePulse(cur^.Position.X, cur^.Position.Y, cur^.Bearing); {Erase pulse}
                                           cur^.Position.X := cur^.Position.X + cos(cur^.Bearing.theta) * cur^.Bearing.mag;
                                           cur^.Position.Y := cur^.Position.Y + sin(cur^.Bearing.theta) * cur^.Bearing.mag;
                                           if cur^.Position.X > 640 then         {If off screen, modify coords}
                                              cur^.Position.X := cur^.Position.X - 640
                                           else if cur^.Position.X <= 0 then
                                                cur^.Position.X := cur^.Position.X + 640;
                                           if cur^.Position.Y > MaxGameY then
                                              cur^.Position.Y := cur^.Position.Y - MaxGameY
                                           else if cur^.Position.Y <= 0 then
                                                cur^.Position.Y := cur^.Position.Y + MaxGameY;
                                           cur^.range := cur^.range - 1;
                                           if (cur^.range >= 0) then {If we must draw it}
                                              DrawLaserpulsePulse(cur^.Position.X, cur^.Position.Y, cur^.Bearing) {Draw pulse}
                                           else
                                               begin {else}
                                                    del := cur;
                                                    cur := cur^.prev;
                                                    cur^.next := del^.next;
                                                    cur^.next^.prev := cur;
                                                    dispose(del)
                                               end {else}
                                      end {if}
                              end; {LASERPULSE_PULSE}
                     SPE_OFFSET + AAMINE : if EShips^.next = EShips then {If there are no enemies}
                                              if cur^.move + AAMINE_MOVE_DELAY <= curtime then {If we must move}
                                                 begin {if}
                                                      cur^.move := curtime;
                                                      EraseAAMine(cur^.Position.X, cur^.Position.Y); {Erase it}
                                                      cur^.Position.X := cur^.Position.X +
                                                                         cos(cur^.Bearing.theta) * cur^.Bearing.mag;
                                                      cur^.Position.Y := cur^.Position.Y +
                                                                         sin(cur^.Bearing.theta) * cur^.Bearing.mag;
                                                      DrawAAMine(cur^.Position.X, cur^.Position.Y); {Draw mine}
                                                      if cur^.Position.X > 645 then                 {If off screen then modify}
                                                          cur^.Position.X := cur^.Position.X - 645
                                                      else if cur^.Position.X <= -8 then
                                                           cur^.Position.X := cur^.Position.X + 645;
                                                      if cur^.Position.Y > MaxGameY + 5 then
                                                         cur^.Position.Y := cur^.Position.Y - MaxGameY
                                                      else if cur^.Position.Y <= -8 then
                                                           cur^.Position.Y := cur^.Position.Y + MaxGameY
                                                 end {if}
                                              else {do nothing}
                                           else if cur^.move + AAMINE_MOVE_DELAY <= curtime then
                                             {If there are enemies and we must move}
                                                begin {else}
                                                    cur^.move := curtime;
                                                    EraseAAMine(cur^.Position.X, cur^.Position.Y); {Erase mine}
                                                    cur^.Position.X := cur^.Position.X +
                                                                       cos(cur^.Bearing.theta) * cur^.Bearing.mag;
                                                    cur^.Position.Y := cur^.Position.Y +
                                                                       sin(cur^.Bearing.theta) * cur^.Bearing.mag;
                                                    DrawAAMine(cur^.Position.X, cur^.Position.Y); {Draw mine}
                                                    if cur^.Position.X > 645 then                    {If off screen, modify}
                                                       cur^.Position.X := cur^.Position.X - 645
                                                    else if cur^.Position.X <= -8 then
                                                         cur^.Position.X := cur^.Position.X + 645;
                                                    if cur^.Position.Y > MaxGameY + 5 then
                                                       cur^.Position.Y := cur^.Position.Y - MaxGameY
                                                    else if cur^.Position.Y <= -8 then
                                                         cur^.Position.Y := cur^.Position.Y + MaxGameY;
                                                    theta := GetTheta(cur^.Position, EShips^.next^.Position);
                                                      {Find angle to target}
                                                    AddVecs(cur^.Bearing.theta, cur^.Bearing.mag, theta,
                                                            HOMING_VELOCITY, cur^.Bearing); {Modify motion vector}
                                                    if cur^.Bearing.mag > AA_MINE_MAX_VELOCITY then {Check for max. speed}
                                                       cur^.Bearing.mag := AA_MINE_MAX_VELOCITY;
                                                    if ((abs(cur^.Position.X - EShips^.next^.Position.X) < AA_MINE_RADIUS +
                                                        ENEMY_RADIUS)) and
                                                       ((abs(cur^.Position.Y - EShips^.next^.Position.Y) < AA_MINE_RADIUS +
                                                        ENEMY_RADIUS)) then {If hit target}
                                                          begin {if}
                                                               EraseScoreMeter(Pilot.Points); {Modify dashboard}
                                                               Pilot.Points := Pilot.Points + GUNSHIP_1_POINT_VALUE;
                                                                 {Modify points}
                                                               CheckForNewLife(Pilot); {Check to see if you got a new life}
                                                               DrawScoreMeter(Pilot.Points); {Modify dashboard}

                                                               if SoundPlaying(EXP_SND) then {If explosion playing}
                                                                  StopSound(EXP_SND); {Stop it}
                                                               StartSound(Sound[EXP_SND], EXP_SND, false); {Start explosion}

                                                               del2 := EShips^.next; {Delete enemy ship}
                                                               EShips^.next := del2^.next;
                                                               EShips^.next^.prev := EShips;
                                                               dispose(del2);
                                                               setfillstyle(SolidFill, LightRed); {Make large explosion}
                                                               setcolor(LightRed);
                                                               fillellipse(round(cur^.Position.X),
                                                                           round(cur^.Position.Y), 60, 60);
                                                               delay(50); {wait}
                                                               setfillstyle(SolidFill, Black); {Erase large explosion}
                                                               setcolor(Black);
                                                               fillellipse(round(cur^.Position.X),
                                                                           round(cur^.Position.Y), 60, 60);
                                                               del := cur; {Delete mine}
                                                               cur := cur^.prev;
                                                               cur^.next := del^.next;
                                                               cur^.next^.prev := cur;
                                                               dispose(del)
                                                          end {if}
                                               end {SPE_OFFSET + AAMINE}
                end; {case}
                cur := cur^.next
           end {while}
end; {AnimateP}

procedure AnimateG(var Goods : GoodPtr; var You : YShipType; var Pilot : Person; var dead : boolean);

{Pre : "Goods" and "You" and "Pilot" are defined.
 Post: All variables are modified if necessary.  Goodies moved if necessary}

var
   cur, del : GoodPtr; {Pointers}
   hr, mn, sc, sc100, curtime : longint; {Time}
   NotHere : boolean; {flag}

begin {AnimateG}
     cur := Goods^.next;
     pGetTime(hr, mn, sc, sc100); {get the time}
     curtime := 360000 * hr + 6000 * mn + 100 * sc + sc100; {Init counter}
     while cur <> Goods do {Loop through all goodies}
           begin {while}
                case cur^.what of {Branch based on what the goody is}
                     CASH : begin {if it is money}
                                 nothere := false;
                                 if curtime >= cur^.move + CASH_MOVE_DELAY then {If it must move}
                                    begin {if}
                                         with cur^.Position do {modify position}
                                              begin {with}
                                                   EraseCash(X, Y); {Erase the money}
                                                   X := X + cur^.Bearing.mag * cos(cur^.Bearing.theta);
                                                   Y := Y + cur^.Bearing.mag * sin(cur^.Bearing.theta);
                                                   if X > 642 then           {If off screen, then modify}
                                                      X := 0
                                                   else if X < -2 then
                                                        X := 640;
                                                   if Y > MaxGameY + 2 then
                                                      Y := 0
                                                   else if Y < -2 then
                                                        Y := MaxGameY;
                                                   if curtime <= cur^.stay then {If it is not gone yet}
                                                      DrawCash(X, Y) {Draw the money}
                                                   else
                                                       begin {else}
                                                            del := cur;             {Unlink it}
                                                            cur := cur^.prev;
                                                            cur^.next := del^.next;
                                                            cur^.next^.prev := cur;
                                                            dispose(del);
                                                            nothere := true {Not here anymore}
                                                       end {else}
                                              end; {with}
                                         cur^.move := curtime
                                    end; {if}
                                 if not(nothere) and (dist(cur^.Position.X, cur^.Position.Y,
                                    You.Position.X, You.Position.Y) <= CASH_RADIUS + 10) then {If you got it!}
                                       begin {if}
                                            if SoundPlaying(MONEY_NOISE) then {If money noise playing}
                                               StopSound(MONEY_NOISE); {stop it}
                                            StartSound(Sound[MONEY_NOISE], MONEY_NOISE, false); {start it}
                                            EraseCashMeter(Pilot.Money);   {Modify dashboard}
                                            EraseScoreMeter(Pilot.Points); {       "        }
                                            Pilot.Money := Pilot.Money + CASH_VALUE; {Give you money}
                                            Pilot.Points := Pilot.Points + CASH_POINT_VALUE; {Give you points}
                                            CheckForNewLife(Pilot); {Check to see if you got a new life}
                                            DrawCashMeter(Pilot.Money);   {Modify dashboard}
                                            DrawScoreMeter(Pilot.Points); {Modify dashboard}
                                            EraseCash(cur^.Position.X, cur^.Position.Y); {Erase cash}
                                            DrawHalo(You.Position.X, You.Position.Y, Green); {Draw halo}
                                            delay(10); {wait}
                                            EraseHalo(You.Position.X, You.Position.Y); {erase halo}
                                            DrawYShip(You.Position.X, You.Position.Y, You.Aim); {Draw your ship again}
                                            del := cur; {delete cash}
                                            cur := cur^.prev;
                                            cur^.next := del^.next;
                                            cur^.next^.prev := cur;
                                            dispose(del)
                                       end {if}
                            end; {CASH}
                     SHIELD : begin {if it is a shield replenishment}
                                   nothere := false;
                                   if curtime >= cur^.move + SHIELD_MOVE_DELAY then {If it must move}
                                      begin {if}
                                           with cur^.Position do {Modify its position}
                                                begin {with}
                                                     EraseShield(X, Y); {Erase it}
                                                     X := X + cur^.Bearing.mag * cos(cur^.Bearing.theta);
                                                     Y := Y + cur^.Bearing.mag * sin(cur^.Bearing.theta);
                                                     if X > 642 then           {If off screen, then modify}
                                                        X := 0
                                                     else if X < -2 then
                                                          X := 640;
                                                     if Y > MaxGameY + 2 then
                                                        Y := 0
                                                     else if Y < -2 then
                                                          Y := MaxGameY;
                                                     if curtime <= cur^.stay then {If it's still here}
                                                        DrawShield(X, Y) {Draw shield}
                                                     else
                                                         begin {else}
                                                              del := cur;             {Unlink it}
                                                              cur := cur^.prev;
                                                              cur^.next := del^.next;
                                                              cur^.next^.prev := cur;
                                                              dispose(del);
                                                              nothere := true {Not here anymore}
                                                         end {else}
                                                end; {with}
                                           cur^.move := curtime
                                      end; {if}
                                   if not(nothere) and (dist(cur^.Position.X, cur^.Position.Y, {If you got it}
                                      You.Position.X, You.Position.Y) <= SHIELD_RADIUS + 10) then
                                         begin {if}
                                              if SoundPlaying(SHIELD_NOISE) then {If shield sound playing}
                                                 StopSound(SHIELD_NOISE); {stop it}
                                              StartSound(Sound[SHIELD_NOISE], SHIELD_NOISE, false); {start it}
                                              EraseScoreMeter(Pilot.Points); {Update dashboard}
                                              Pilot.Points := Pilot.Points + SHIELD_POINT_VALUE; {Update points}
                                              CheckForNewLife(Pilot); {Check to see if you got a new life}
                                              DrawScoreMeter(Pilot.Points);  {Update dashboard}
                                              You.Shields := You.Shields + SHIELD_VALUE; {Update shields}
                                              if You.Shields > Pilot.MaxShields then {If your shields are above max, then
                                                                                      make them the max.}
                                                 You.Shields := Pilot.MaxShields;
                                              EraseShieldMeter; {Update dashboard}
                                              DrawShieldMeter(You.Shields, Pilot.MaxShields); {Update dashboard}
                                              EraseShield(cur^.Position.X, cur^.Position.Y); {Erase shield bonus}
                                              DrawHalo(You.Position.X, You.Position.Y, Blue); {Draw a halo}
                                              delay(10); {wait}
                                              EraseHalo(You.Position.X, You.Position.Y); {erase the halo}
                                              DrawYShip(You.Position.X, You.Position.Y, You.Aim); {draw your ship again}
                                              del := cur; {unlink element from list}
                                              cur := cur^.prev;
                                              cur^.next := del^.next;
                                              cur^.next^.prev := cur;
                                              dispose(del)
                                         end {if}
                              end {SHIELD}
                end; {case}
                cur := cur^.next
           end {while}
end; {AnimateG}

procedure AnimateE(var Enemies : EShipPtr; var You : YShipType; var Projs : ProjPtr; var bak2shop, dead : boolean);

{Pre:  "Enemies", "You", "Projs" defined.
 Post: If enemies are to move, then they move.  If they get hit, they disappear.
       If you crash into them, then they may disappear.}

var
   cur, del : EShipPtr; {Pointers}
   hr, mn, sc, sc100, curtime : longint; {Time}
   hit, hitbullet : boolean; {Flags}
   del2 : ProjPtr; {deletion ptrs.}
   bul : ProjPtr; {deletion ptrs.}

begin {AnimateE}
     pGetTime(hr, mn, sc, sc100); {Get the time}
     curtime := 360000 * hr + 6000 * mn + 100 * sc + sc100; {Fix the time}
     cur := Enemies^.next;
     while cur <> Enemies do {Loop through all enemies}
           begin {while}
                case cur^.what of {Branch based on what the enemy is}
                     GUNSHIP_1 : if not((cur^.Position.X > -2) and (cur^.Position.X < 642) and
                                        (cur^.Position.Y > -2) and (cur^.Position.Y < MaxGameY)) then {If it is on screen}
                                           begin {if}
                                                EraseGunship1(cur^.Position.X, cur^.Position.Y); {Erase the gunship}
                                                del := cur; {dispose of the element}
                                                cur := cur^.prev;
                                                cur^.next := del^.next;
                                                cur^.next^.prev := cur;
                                                dispose(del)
                                           end {if}
                                 else begin {else}
                                      hit := false;
                                      if curtime >= cur^.mov + GUNSHIP_1_MOVE_DELAY then {If time to move}
                                         begin {if}
                                              EraseGunship1(cur^.Position.X, cur^.Position.Y); {erase the gunship}
                                              cur^.Position.X := cur^.Position.X + cos(cur^.Bearing.theta) * cur^.Bearing.mag;
                                              cur^.Position.Y := cur^.Position.Y + sin(cur^.Bearing.theta) * cur^.Bearing.mag;
                                              DrawGunship1(cur^.Position.X, cur^.Position.Y); {Draw the gunship}
                                              cur^.mov := curtime;
                                              if (abs(cur^.Position.X - You.Position.X) < GUNSHIP_1_RADIUS + 10) and
                                                 (abs(cur^.Position.Y - You.Position.Y) < GUNSHIP_1_RADIUS + 10) then
                                                   {If you hit it}
                                                    begin {if}
                                                         EraseScoreMeter(Pilot.Points); {Update dashboard}
                                                         Pilot.Points := Pilot.Points + GUNSHIP_1_POINT_VALUE;
                                                         CheckForNewLife(Pilot); {Check to see if you got a new life}
                                                         DrawScoreMeter(Pilot.Points); {Update the dashboard}
                                                         if curtime >= You.InvulnerableStay then {If you are not invulnerable}
                                                            You.Shields := You.Shields - cur^.Shields; {Update your shields}
                                                         EraseShieldMeter; {Update meter}
                                                         DrawShieldMeter(You.Shields, Pilot.MaxShields); {Update your meter}
                                                         if You.Shields <= 0 then {If you are out of shields}
                                                            if Pilot.Lives = 0 then {If you are out of lives}
                                                               dead := true
                                                            else
                                                                begin {else}
                                                                     EraseYShip(You.Position.X, You.Position.Y, You.Aim);
                                                                       {erase your ship}
                                                                     Dec(Pilot.Lives); {One less life}
                                                                     FixYou(You, Pilot) {Re-init your ship}
                                                                end {else}
                                                         else
                                                             begin {else}
                                                                  if SoundPlaying(EXP_SND) then {If exp. playing}
                                                                     StopSound(EXP_SND); {stop it}
                                                                  StartSound(Sound[EXP_SND], EXP_SND, false); {start it}
                                                                  EraseGunship1(cur^.Position.X, cur^.Position.Y);
                                                                    {erase gunship}
                                                                  setfillstyle(solidfill, lightred); {draw explosion}
                                                                  setcolor(lightred);
                                                                  fillellipse(round(cur^.Position.X),
                                                                              round(cur^.Position.Y), 30, 30);
                                                                  delay(10); {wait}
                                                                  setcolor(black); {erase explosion}
                                                                  setfillstyle(solidfill, black);
                                                                  fillellipse(round(cur^.Position.X),
                                                                              round(cur^.Position.Y), 30, 30);
                                                                  del := cur; {remove element}
                                                                  cur := cur^.prev;
                                                                  cur^.next := del^.next;
                                                                  cur^.next^.prev := cur;
                                                                  dispose(del);
                                                                  hit := true
                                                             end {else}
                                                    end {if}
                                         end; {if}
                                      hitbullet := false;
                                      bul := Projs^.next;
                                      while not(hitbullet) and (bul <> Projs) do {Check to see if you it hit a bullet}
                                            begin {while}
                                                 if (bul^.what = BULLET) or (bul^.what = POWERGUN_BULLET) then
                                                  {If it's a bullet}
                                                    if (abs(cur^.Position.X - bul^.Position.X) < GUNSHIP_1_RADIUS + 2) and
                                                       (abs(cur^.Position.Y - bul^.Position.Y) < GUNSHIP_1_RADIUS + 2) then
                                                         {If it hit a bullet}
                                                          begin {if}
                                                               if SoundPlaying(EXP_SND) then {if exp. playing}
                                                                  StopSound(EXP_SND); {stop it}
                                                               StartSound(Sound[EXP_SND], EXP_SND, false); {start it}
                                                               EraseScoreMeter(Pilot.Points); {erase score meter}
                                                               Pilot.Points := Pilot.Points + GUNSHIP_1_POINT_VALUE;
                                                                 {Update points}
                                                               CheckForNewLife(Pilot); {Check to see if you got a new life}
                                                               DrawScoreMeter(Pilot.Points); {Update score meter}
                                                               EraseGunship1(cur^.Position.X, cur^.Position.Y); {erase gunship}
                                                               setfillstyle(solidfill, lightred); {Draw explosion}
                                                               setcolor(lightred);
                                                               fillellipse(round(cur^.Position.X),
                                                                           round(cur^.Position.Y), 30, 30);
                                                               delay(10); {wait}
                                                               setcolor(black); {Erase explosion}
                                                               setfillstyle(solidfill, black);
                                                               fillellipse(round(cur^.Position.X),
                                                                           round(cur^.Position.Y), 30, 30);
                                                               del := cur; {Delete element on LL (Linked list}
                                                               cur := cur^.prev;
                                                               cur^.next := del^.next;
                                                               cur^.next^.prev := cur;
                                                               dispose(del);
                                                               hitbullet := true;
                                                               if bul^.what = BULLET then {If the proj. was a bullet, not a
                                                                                           power bullet}
                                                                  begin {if}
                                                                       del2 := bul; {Unlink it}
                                                                       bul := bul^.prev;
                                                                       bul^.next := del2^.next;
                                                                       bul^.next^.prev := bul;
                                                                       dispose(del2)
                                                                  end {if}
                                                          end; {if}
                                                 if (bul^.what = LASERPULSE_PULSE) then {If it's a laser pulse}
                                                    if (abs(cur^.Position.X - bul^.Position.X) < GUNSHIP_1_RADIUS +
                                                        LASERPULSE_PULSE_LENGTH) and
                                                       (abs(cur^.Position.Y - bul^.Position.Y) < GUNSHIP_1_RADIUS +
                                                        LASERPULSE_PULSE_LENGTH) then {If it hit}
                                                          begin {if}
                                                               if SoundPlaying(EXP_SND) then {If explosion playing}
                                                                  StopSound(EXP_SND); {stop it}
                                                               StartSound(Sound[EXP_SND], EXP_SND, false); {start it}
                                                               EraseScoreMeter(Pilot.Points); {Update score meter}
                                                               Pilot.Points := Pilot.Points + GUNSHIP_1_POINT_VALUE;
                                                                 {Update points}
                                                               CheckForNewLife(Pilot); {Check to see if you got a new life}
                                                               DrawScoreMeter(Pilot.Points); {Update score meter}
                                                               EraseGunship1(cur^.Position.X, cur^.Position.Y); {erase gunship}
                                                               setfillstyle(solidfill, lightred); {draw explosion}
                                                               setcolor(lightred);
                                                               fillellipse(round(cur^.Position.X),
                                                                           round(cur^.Position.Y), 30, 30);
                                                               delay(10); {wait}
                                                               setcolor(black); {erase explosion}
                                                               setfillstyle(solidfill, black);
                                                               fillellipse(round(cur^.Position.X),
                                                                           round(cur^.Position.Y), 30, 30);
                                                               del := cur; {Unlink element}
                                                               cur := cur^.prev;
                                                               cur^.next := del^.next;
                                                               cur^.next^.prev := cur;
                                                               dispose(del);
                                                               hitbullet := true
                                                          end; {if}
                                                 bul := bul^.next
                                            end; {while}
                                      if not(hit) and not(hitbullet) and (curtime >= cur^.shoot + GUNSHIP_1_SHOOT_DELAY) then
                                       {If it's not gone, and it's going to shoot}
                                         begin {if}
                                              fireEbullet(cur, Projs, You); {shoot}
                                              cur^.shoot := curtime
                                         end {if}
                                 end {else}
                end; {case}
                cur := cur^.next
           end {while}
end; {AnimateE}

procedure AnimateA(var Asts : AstPtr; var You : YShipType; var Goods : GoodPtr;
                   var bak2shop, dead : boolean; var Projs : ProjPtr);

{Pre:  All arguments are defined.
 Post: All asteroids move, they may be destroyed, and they may release goodies}

var
   cur, Last : AstPtr; {pointers}
   curtime, hr, mn, sc, sc100 : longint; {time}
   bul, del : ProjPtr; {deletion ptrs. and traversion ptrs.}
   hitbullet : boolean; {flag}

   procedure BreakUp(var Ast : AstPtr; var Goods : GoodPtr);

   {Pre:  Ast is defined and Goods is defined.
    Post: The asteroid breaks up if it is a large one, and explodes if it is
          a small one.  Goodies may be released.}

   var
      del, ast2, ast3 : AstPtr; {Insertion and deletion ptrs}
      NumGoods : integer; {counter}
      ctr : integer; {counter}
      ins, GNext : GoodPtr; {traversion ptrs. for goodies}

   begin {BreakUp}
        setcolor(LightRed); {Make explosion}
        setfillstyle(SolidFill, LightRed);
        fillellipse(round(Ast^.Position.X), round(Ast^.Position.Y), 30, 30);
        if SoundPlaying(EXP_SND) then {If explosion playing}
           StopSound(EXP_SND); {stop it}
        StartSound(Sound[EXP_SND], EXP_SND, false); {start it}
        delay(5); {wait}
        setcolor(Black); {erase explosion}
        setfillstyle(solidfill, black);
        fillellipse(round(Ast^.Position.X), round(Ast^.Position.Y), 30, 30);
        if Ast^.Size = SMALL then {If the asteroid was small}
           begin {if}
                del := Ast; {Unlink it}
                Ast := Ast^.prev;
                Ast^.next := del^.next;
                Ast^.next^.prev := Ast;
                dispose(del)
           end {if}
        else {it was large}
            begin {else}
                 Ast^.Size := Ast^.Size - 1;
                 new(Ast2); {Add new asteroid}
                 with Ast2^ do
                      begin {with}
                           Size := Ast^.Size;
                           Position.X := Ast^.Position.X + random(21) - 10;
                           Position.Y := Ast^.Position.Y + random(21) - 10;
                           Bearing.theta := random(360) * Pi/180;
                           Bearing.mag := random(3) + 1;
                           move := 0;
                           prev := Ast
                      end; {with}
                 new(Ast3); {Add another asteroid}
                 with Ast3^ do
                      begin {with}
                           Size := Ast^.Size;
                           Position.X := Ast^.Position.X + random(21) - 10;
                           Position.Y := Ast^.Position.Y + random(21) - 10;
                           Bearing.theta := random(360) * Pi/180;
                           Bearing.mag := random(3) + 1;
                           move := 0;
                           prev := Ast2;
                           next := Ast^.next
                      end; {with}
                 Ast2^.next := Ast3; {Link 'em up}
                 Ast^.next^.prev := Ast3;
                 Ast^.next := Ast2;
                 NumGoods := 0; {Init counter}
                 while random(GoodChance) = 0 do {random number of goodies}
                       NumGoods := NumGoods + 1;
                 ins := Goods; {Insertion ptr to add multiple goodies}
                 GNext := Goods^.next;
                 for ctr := 1 to NumGoods do {Add NUMGOODS goodies}
                     begin {for}
                          new(ins^.next);
                          with ins^.next^ do
                               begin {with}
                                    what := random(2);
                                    Bearing.theta := random(360) * Pi/180;
                                    Bearing.mag := random(3) + 1;
                                    Position.X := Ast^.Position.X + random(21) - 10;
                                    Position.Y := Ast^.Position.Y + random(21) - 10;
                                    move := 0;
                                    pGetTime(hr, mn, sc, sc100);
                                    curtime := 360000 * hr + 6000 * mn + 100 * sc + sc100;
                                    stay := curtime + GOODY_STAY_TIME;
                                    prev := ins;
                                    next := GNext
                               end; {with}
                          GNext^.prev := ins^.next; {Link}
                          ins := ins^.next
                     end {for}
            end {else}
   end; {BreakUp}

begin {AnimateA}
     pGetTime(hr, mn, sc, sc100); {Get time}
     curtime := 360000 * hr + 6000 * mn + 100 * sc + sc100; {Init time ctr}
     cur := Asts^.next;
     while cur <> Asts do {Animate all asteroids}
           begin {while}
                with cur^ do
                     begin {with}
                          if move + AST_MOVE_DELAY <= curtime then {If ast. moves}
                             begin {if}
                                  move := curtime;
                                  EraseAst(Position.X, Position.Y, Size); {erase asteroid}
                                  Position.X := Position.X + cos(Bearing.theta) * Bearing.mag;
                                  Position.Y := Position.Y + sin(Bearing.theta) * Bearing.mag;
                                  if Position.X > 640 then {If off screen, modify}
                                     Position.X := Position.X - 640
                                  else if Position.X <= 0 then
                                       Position.X := Position.X + 640;
                                  if Position.Y > MaxGameY then
                                     Position.Y := Position.Y - MaxGameY
                                  else if Position.Y <= 0 then
                                       Position.Y := Position.Y + MaxGameY;
                                  DrawAst(Position.X, Position.Y, Size) {draw asteroid}
                             end; {if}
                          hitbullet := false;
                          bul := Projs^.next;
                          while not(hitbullet) and (bul <> Projs) do {Check to see if ast. was hit}
                                begin {while}
                                     if (bul^.what = BULLET) or (bul^.what = POWERGUN_BULLET) then {If it is a bullet}
                                        if (abs(Position.X - bul^.Position.X) < AST_RADIUS * Size + 2) and
                                           (abs(Position.Y - bul^.Position.Y) < AST_RADIUS * Size + 2) then {If it hit}
                                              begin {if}
                                                   EraseScoreMeter(Pilot.Points); {Update score meter}
                                                   Pilot.Points := Pilot.Points + ROCK_POINT_VALUE * Size; {Update points}
                                                   CheckForNewLife(Pilot); {Check to see if you got a new life}
                                                   DrawScoreMeter(Pilot.Points); {Update score meter}
                                                   setfillstyle(solidfill, black); {Erase asteroid}
                                                   setcolor(black);
                                                   fillellipse(round(Position.X), round(Position.Y),
                                                               AST_RADIUS * Size + 1, AST_RADIUS * Size + 1);
                                                   BreakUp(cur, Goods); {Make asteroid break up}
                                                   hitbullet := true;
                                                   if bul^.what = BULLET then {Check if it was a normal bullet}
                                                      begin {if}
                                                           del := bul;
                                                           bul := bul^.prev;
                                                           bul^.next := del^.next;
                                                           bul^.next^.prev := bul;
                                                           dispose(del)
                                                      end {if}
                                              end; {if}
                                     if (bul^.what = LASERPULSE_PULSE) then {If it's a laser pulse}
                                         if (abs(Position.X - bul^.Position.X) < AST_RADIUS * Size +
                                             LASERPULSE_PULSE_LENGTH) and
                                            (abs(Position.Y - bul^.Position.Y) < AST_RADIUS * Size +
                                             LASERPULSE_PULSE_LENGTH) then {If it hit}
                                              begin {if}
                                                   EraseScoreMeter(Pilot.Points); {erase score meter}
                                                   Pilot.Points := Pilot.Points + ROCK_POINT_VALUE * Size; {Update score}
                                                   CheckForNewLife(Pilot); {Check to see if you got a new life}
                                                   DrawScoreMeter(Pilot.Points); {draw score meter}
                                                   setfillstyle(solidfill, black); {erase asteroid}
                                                   setcolor(black);
                                                   fillellipse(round(Position.X), round(Position.Y),
                                                               AST_RADIUS * Size + 1, AST_RADIUS * Size + 1);
                                                   BreakUp(cur, Goods); {break it up}
                                                   hitbullet := true
                                              end; {if}
                                        bul := bul^.next
                                end; {while}
                          if not(hitbullet) and {If it hasn't been destroyed}
                             (abs(Position.X - You.Position.X) < AST_RADIUS * Size + 10) and
                             (abs(Position.Y - You.Position.Y) < AST_RADIUS * Size + 10) then {If it hit you}
                                begin {if}
                                     EraseScoreMeter(Pilot.Points); {erase score meter}
                                     Pilot.Points := Pilot.Points + ROCK_POINT_VALUE * Size; {Update points}
                                     CheckForNewLife(Pilot); {Check to see if you got a new life}
                                     DrawScoreMeter(Pilot.Points); {draw score meter}
                                     setfillstyle(solidfill, black); {erase asteroid}
                                     setcolor(black);
                                     fillellipse(round(Position.X), round(Position.Y),
                                                 AST_RADIUS * Size + 1, AST_RADIUS * Size + 1);
                                     if curtime >= You.InvulnerableStay then {If you aren't invulnerable}
                                        You.Shields := You.Shields - AST_DAMAGE * Size; {Lower your shields}
                                     EraseShieldMeter; {erase meter}
                                     DrawShieldMeter(You.Shields, Pilot.MaxShields); {draw meter}
                                     if You.Shields <= 0 then {If no shields left}
                                        if Pilot.Lives = 0 then {and no lives left}
                                           begin {if}
                                                dead := true; {You are dead}
                                                exit {Leave}
                                           end {if}
                                        else {Lives left}
                                            begin {else}
                                                 EraseYShip(You.Position.X, You.Position.Y, You.Aim); {erase your ship}
                                                 Dec(Pilot.Lives); {Lose one life}
                                                 FixYou(You, Pilot) {Re-init. your ship}
                                            end {else}
                                     else {still shields}
                                         BreakUp(cur, Goods) {Break it up}
                                end {if}
                     end; {case}
                cur := cur^.next
           end; {while}
     if Asts^.next = Asts then {If no asteroids left}
        bak2shop := true {go back to shop}
end; {AnimateA}

procedure process(var You : YShipType; var pilot : Person; var Projs : ProjPtr; var bak2menu : boolean);

{Pre:  All arguments defined.
 Post: Keyboard commands will be processed.}

var
   hr, mn, sc, sc100, curtime : longint; {time}
   NumSpecs, ctr : integer; {counters}

begin {process}
     pGetTime(hr, mn, sc, sc100); {get time}
     curtime := 360000 * hr + 6000 * mn + 100 * sc + sc100; {init counter}
     if CtrlFlag and (Pilot.Gun in [1, 2]) and ((curtime >= You.shoot + Y_CANNON_REFIRE_DELAY) or
        Pilot.Automatic and (curtime >= You.shoot + Y_AUTOMATIC_DELAY)) then
         {If you are to shoot a normal bullet}
           begin {if}
                You.shoot := curtime;
                ErasePower; {erase power bar}
                if You.Power >= BULLET_POWER then {enough power}
                   firebullet(You, Projs); {shoot}
                DrawPower(You.Power, Pilot.GeneratorType * 20) {update power}
           end {if}
     else if CtrlFlag and (Pilot.Gun = 3) and (curtime >= You.shoot + Y_POWERGUN_REFIRE_DELAY) then
      {If you are to shoot a power bullet}
          begin {else if}
               You.shoot := curtime;
               ErasePower; {erase power bar}
               if You.Power >= POWERGUN_POWER then {If enough power}
                  firepowergun(You, Projs); {shoot}
               DrawPower(You.Power, Pilot.GeneratorType * 20) {update power}
          end {else if}
     else if CtrlFlag and (Pilot.Gun = 4) and (curtime >= You.shoot + Y_LASERPULSE_REFIRE_DELAY) then
      {If you are shooting a laser pulse}
          begin {else if}
               You.shoot := curtime;
               ErasePower; {erase power bar}
               if You.Power >= LASERPULSE_POWER then {if enough power}
                  firelaserpulse(You, Projs); {shoot}
               DrawPower(You.Power, Pilot.GeneratorType * 20) {draw power bar}
          end {else if}
      else if CtrlFlag and (Pilot.Gun = 5) and (curtime >= You.shoot + Y_LASERPULSE_REFIRE_DELAY) then
      {If you are shooting a 2X laser pulse}
          begin {else if}
               You.shoot := curtime;
               ErasePower; {erase power bar}
               if You.Power >= LASERPULSE_POWER then {if enough power}
                  fire2laserpulse(You, Projs); {shoot}
               DrawPower(You.Power, Pilot.GeneratorType * 20) {draw power bar}
          end; {else if}
     if UpFlag and {If accelerating}
        (curtime >= You.acc + Pilot.AccDelay) and
        (You.bearing.mag < Y_MAX_VELOCITY * Pilot.Thruster) and
        (You.Power >= ACC_PWR) then
           begin {if}
                You.acc := curtime;
                AddVecs(You.bearing.theta, You.bearing.mag, You.aim, Y_ACC_INCREMENT, You.bearing); {Accelerate}
                if You.bearing.mag > Y_MAX_VELOCITY * Pilot.Thruster then {You cannot go above max. velocity}
                   You.bearing.mag := Y_MAX_VELOCITY * Pilot.Thruster;
                ErasePower; {Update power bar}
                You.Power := You.Power - ACC_PWR; {Update power}
                DrawPower(You.Power, Pilot.GeneratorType * 20) {draw power bar}
           end; {if}
     if DownFlag and {If braking}
        (curtime >= You.acc + Pilot.AccDelay) and
        (You.bearing.mag > 0) then
           begin {if}
                You.acc := curtime;
                You.bearing.mag := You.bearing.mag - Y_ACC_INCREMENT; {slow down}
                if You.bearing.mag < 0 then
                   You.bearing.mag := 0
           end; {if}
     if RightFlag and (curtime >= You.turn + Y_TURN_DELAY) then {If turning}
        begin {if}
             You.turn := curtime;
             You.aim := You.aim + Y_TURN_ANGLE
        end; {if}
     if LeftFlag and (curtime >= You.turn + Y_TURN_DELAY) then {If turning}
        begin {if}
             You.turn := curtime;
             You.aim := You.aim - Y_TURN_ANGLE
        end; {if}
     if EscFlag then {If pausing}
        begin {if}
             while not(TabFlag or CtrlFlag) do {Wait until tab is pressed to resume,
                                                or ctrl is pressed to exit}
                   ; {do nothing}
             if CtrlFlag then {If quitting}
                bak2menu := true {go back to menu}
        end; {if}
     NumSpecs := 0; {counter to zero}
     for ctr := 1 to NumSpecials do {find number of special weapons posessed}
         if Pilot.Specials[ctr] > 0 then {If there is more than one of a category}
            NumSpecs := NumSpecs + 1;
     if (Shifted and kbLeftShift <> 0) and {If switching specials}
       (You.specswitch + SWITCH_DELAY <= curtime) then
        if NumSpecs > 1 then {If more than one special}
           begin {if}
                EraseSpecialsMeter(Pilot); {erase specials meter}
                ctr := Pilot.SpePosition; {ctr to current special position}
                ctr := ctr - 1;
                if ctr = 0 then {If decremented to zero}
                   ctr := NumSpecials; {Loop back around}
                while Pilot.Specials[ctr] = 0 do {get next special position}
                      begin {while}
                           ctr := ctr - 1;
                           if ctr = 0 then {If decremented to zero}
                              ctr := NumSpecials {Loop around}
                      end; {while}
                Pilot.SpePosition := ctr; {New specials position}
                DrawSpecialsMeter(Pilot); {draw specials meter}
                You.specswitch := curtime
           end; {if}
     if (Shifted and kbRightShift <> 0) and {If switching}
       (You.specswitch + SWITCH_DELAY <= curtime) then
        if NumSpecs > 1 then {Same procedure as above segment, except cycling
                              in the opposite direction}
           begin {if}
                EraseSpecialsMeter(Pilot);
                ctr := Pilot.SpePosition;
                ctr := ctr + 1;
                if ctr = NumSpecials + 1 then
                   ctr := 1;
                while Pilot.Specials[ctr] = 0 do
                      begin {while}
                           ctr := ctr + 1;
                           if ctr = NumSpecials + 1 then
                              ctr := 1
                      end; {while}
                Pilot.SpePosition := ctr;
                DrawSpecialsMeter(Pilot);
                You.specswitch := curtime
           end; {if}
      if AltFlag and (curtime >= You.spec + SPECIAL_REFIRE_DELAY) then {If dropping special}
         begin {if}
              EraseSpecialsMeter(Pilot); {erase meter}
              LaunchSpecial(You, Projs, Pilot); {drop special}
              DrawSpecialsMeter(Pilot); {draw meter}
              You.spec := curtime
         end {if}
end; {process}

procedure DeAllocate(var Asts : AstPtr; var Goods : GoodPtr;
                     var Enemies : EShipPtr; var Projs : ProjPtr);

{Pre:  All arguments are defined.
 Post: All memory is deallocated.}

var
   Rock, delRock : AstPtr; {Deletion and traversion ptrs}
   Good, delGood : GoodPtr; {Deletion and traversion ptrs}
   En, delEn : EShipPtr; {Deletion and traversion ptrs}
   Proj, delProj : ProjPtr; {Deletion and traversion ptrs}

begin {DeAllocate}
     Rock := Asts;
     while Rock^.next <> Rock do {Deallocate all memory}
           begin {while}
                delRock := Rock^.next;
                Rock^.next := Rock^.next^.next;
                Rock^.next^.prev := Rock;
                dispose(delRock)
           end; {while}
     dispose(Rock);

     Good := Goods;
     while Good^.next <> Good do {Deallocate all memory}
           begin {while}
                delGood := Good^.next;
                Good^.next := Good^.next^.next;
                Good^.next^.prev := Good;
                dispose(delGood)
           end; {while}
     dispose(Good);

     En := Enemies;
     while En^.next <> En do {Deallocate all memory}
           begin {while}
                delEn := En^.next;
                En^.next := En^.next^.next;
                En^.next^.prev := En;
                dispose(DelEn)
           end; {while}
     dispose(En);

     Proj := Projs;
     while Proj^.next <> Proj do {Deallocate all memory}
           begin {while}
                DelProj := Proj^.next;
                Proj^.next := Proj^.next^.next;
                Proj^.next^.prev := Proj;
                dispose(delProj)
           end; {while}
     dispose(Proj)
end; {DeAllocate}

procedure shop(var Pilot : Person; var bak2menu : boolean);

{Pre:  None.
 Post: Shopping complete}

var
   Mse : MouseObject; {Mouse}
   OldImage : pointer; {Image ptrs}
   right, mid, left : boolean; {Button flags}
   row, col : integer; {XY counters}
   Leave : boolean; {flag}
   Choice : byte; {choice}
   On : boolean; {flag}

begin {shop}
     Leave := false;
     setviewport(0, 0, 639, 479, true); {viewport to whole screen}
     repeat {do shopping}
           DrawMainRoom; {Draw the main room}
           GetMem(OldImage, ImageSize(200, 450, 440, 480)); {Get memory}
           GetImage(200, 450, 440, 480, OldImage^); {Get image}
           Mse.movecursor(320, 220); {Move cursor to center}
           Mse.showmouse; {show mouse}
           Mse.VertLimit(0, 478); {reset mouse limits}
           Mse.HorizLimit(0, 638);
           Choice := 0;
           On := false;
           repeat {wait until mouse is clicked in proper area}
                 Mse.GetClicks(right, mid, left, row, col); {get mouse status}
                 if not(On) then {If not on an image}
                    putimage(200, 450, OldImage^, COPYPUT); {Replace original image}
                 if OnChoice(row, col, Choice) then {If on a choice}
                    begin {if}
                         On := true;
                         WriteText(Choice) {Put text on bottom of screen}
                    end {if}
                 else
                     On := false {Not on an item}
           until (Choice <> 0) and left;
           Mse.VertLimit(0, 480); {Reset mouse limits}
           Mse.hidemouse; {Hide mouse cursor}
           case Choice of {Branch based on what was clicked.}
                1 : DoUpgrades(Pilot); {Upgrade ship}
                2 : DoSave(Pilot); {save game}
                3 : Leave := true; {Fly mission}
                4 : DoBank(Pilot); {financial}
                5 : Leave := true {quit}
           end; {case}
           FreeMem(OldImage, ImageSize(200, 450, 440, 480)) {Deallocate memory for picture}
     until Leave;
     if Choice = 5 then {If quit chosen}
        bak2menu := true {Go back to menu}
end; {Shop}

procedure Lost;

{Pre:  None.
 Post: "You LOST!" output to screen}

begin {Lost}
     setcolor(LightGreen);
     cleardevice; {Clear screen}
     settextstyle(TriplexFont, HorizDir, 5);
     settextjustify(CenterText, CenterText);
     OutTextXY(320, 100, 'You LOST!'); {Write "You LOST!"}
     delay(1000) {wait for 1 second}
end; {Lost}

procedure Victory;

{Pre:  None.
 Post: "You WON!" output to screen}

begin {Victory}
     setcolor(LightGreen);
     cleardevice;
     settextstyle(TriplexFont, HorizDir, 5);
     settextjustify(CenterText, CenterText);
     OutTextXY(320, 100, 'You WON!'); {Say "You WON!"}
     delay(1000) {wait for 1 sec.}
end; {Victory}

procedure PlayGame(var Pilot : Person);

{Pre:  None.
 Post: Game is played until user decides to stop}

var
   Asts : AstPtr; {Asteroids}
   Goods : GoodPtr; {Goodies}
   Enemies : EShipPtr; {Enemies}
   You : YShipType; {Your ship}
   Projs : ProjPtr; {Projectiles}

   MaxLevel : integer; {Highest level in game}

   LevData : LevelType; {Level data}

   bak2shop, bak2menu, dead, won : boolean; {flags}

   hr, mn, sc, sc100, curtime : longint; {time}
   LastAdd : longint; {time counter}

begin {PlayGame}
     dead := false;
     won := false;
     bak2menu := false;
     Shop(Pilot, bak2menu); {Go to the shop}
     if bak2menu then {If user chose quit}
        exit; {go to main menu}
     repeat {Play multiple games}
           cleardevice; {clear screen}
           Init(Asts, Goods, Enemies, You, Projs, Pilot, MaxLevel, LevData);
            {Init all arguments}
           bak2shop := false;
           DrawMeterBox; {Draw meter box (dashboard)}
           LastAdd := 0;
           repeat {Play level until user dies, quits, or wins}
                 pGetTime(hr, mn, sc, sc100); {Get time}
                 curtime := 360000 * hr + 6000 * mn + 100 * sc + sc100; {Set ctr}
                 AnimateY(You, Pilot, bak2shop, dead); {Animate your ship}
                 AnimateP(Projs, You, Enemies, bak2shop, dead); {Animate projectiles}
                 AnimateG(Goods, You, Pilot, dead); {Animate goodies}
                 AnimateE(Enemies, You, Projs, bak2shop, dead); {Animate enemies}
                 AnimateA(Asts, You, Goods, bak2shop, dead, Projs); {Animate asteroids}
                 if (LevData.AlienLevel > 0) and (curtime > LastAdd + 100) then
                  {If time to add alien}
                    begin {if}
                         if random(LevData.AlienChance) = 0 then {If random enemy to be added}
                            AddAlien(Enemies, random(LevData.AlienLevel)); {Add alien}
                         LastAdd := curtime
                    end; {if}
                 if CtrlFlag or UpFlag or DownFlag or LeftFlag or RightFlag or
                    EscFlag or (Shifted and kbLeftShift <> 0) or
                    (Shifted and kbRightShift <> 0) or AltFlag or TabFlag then
                      {If keyboard input detected}
                       process(You, Pilot, Projs, bak2menu) {Process it}
           until bak2shop or bak2menu or dead;
           DeAllocate(Asts, Goods, Enemies, Projs); {deallocate all memory}
           Pilot.level := Pilot.level + 1;
           if not(bak2menu) and (Pilot.level > MaxLevel) then {If you won}
              won := true;
           if not(bak2menu) and not(dead) and not(won) then {If it's time to shop}
              Shop(Pilot, bak2menu); {Go to shop}
           setviewport(0, 0, 639, 479, true) {reset viewport}
     until won or dead or bak2menu;
     if dead then {If you're dead}
        Lost {go to defeat sequence}
     else if won then {If you won}
          Victory {go to victory sequence}
end; {PlayGame}

procedure NGame(var Pilot : Person);

{Desc: Start new game.
 Pre:  None.
 Post: New game started}

var
   nam : string; {Temp. string}
   err : integer; {Error code}
   ctr : integer; {counter}

begin {NGame}
     NormalOperation := true; {Set keyboard flag}
     repeat {Gets name}
           clearviewport; {Clear screen}
           Fnt.psprintf(5, 100, 8, LightCyan, 1, 'Please enter your name:'); {Ask for name}
           err := Fnt.psflashscanf(nam, 50, 5, 120, 8, Cyan); {Get name}
           Dec(nam[0]) {remove extraneous character}
     until err = 0;
     NormalOperation := false; {Set keyboard flag}
     with Pilot do {Modify pilot}
          begin {with}
               Lives := 3; {3 lives}
               LastLifeUp := 0;
               Gun := 1; {bullet gun}
               Automatic := false; {Not a machine gun}
               for ctr := 1 to NumSpecials do {No specials}
                   Specials[ctr] := 0;
               SpePosition := 0;
               MaxShields := 10; {10 shield units}
               GeneratorType := BasicGenerator; {basic generator}
               RechargeDelay := 5; {slow recharge delay}
               ShieldType := 1; {basic shields}
               Thruster := BasicThruster; {basic thruster}
               AccDelay := 10; {slow acceleration}
               Level := 1; {Low level}
               if TESTING {FIX} then
                  Money := 1000
               else
                   Money := 0; {No cash}
               Points := 0; {No points}
               Name := UpCaseStr(nam) {All caps for name}
          end; {with}
     PlayGame(Pilot) {Play the game}
end; {NGame}

procedure LoadGame(var Pilot : Person);

{Desc: Loads saved game
 Pre:  None.
 Post: Game loaded and played}

type
    LLPtr = ^LLType; {Linked list ptr}
    LLType = record {Linked list node}
                   Data : Person; {Data}
                   prev, next : LLPtr {Links}
             end; {record}

var
   SaveFile : file of Person; {file}
   IniFile, TempFile : text; {files}
   cur, Head, trav, del : LLPtr; {traversion and deletion ptrs}
   Choice : string; {button choice}
   OK, Cancel : boolean; {flags}
   But : OldButtonObject; {buttons}
   Mse : MouseObject; {mouse}
   ctr, num : integer; {ctrs}
   tstr : string; {temp. str}
   pos : integer; {position ctr}
   ch : char; {temp. character}

begin {LoadGame}
     cleardevice; {Clear screen}
     Mse.VertLimit(0, 478); {reset mouse limits}
     Mse.HorizLimit(0, 638);

     assign(IniFile, GetPossPath(IniPaths, IniFileName)); {Get path for .ini file}
     reset(IniFile);
     readln(IniFile, num);
     for ctr := 1 to num do {read past level section}
         readln(IniFile);
     readln(IniFile, num);
     if num = 0 then {If no saved games}
        begin {if}
             cleardevice; {clear screen}
             setcolor(LightRed);
             settextstyle(SmallFont, HorizDir, 6);
             settextjustify(CenterText, CenterText);
             OutTextXY(320, 240, 'NO SAVED GAMES AVAILABLE!'); {Show error message}
             delay(1000); {wait}
             exit {Leave back to main menu}
        end; {if}

     setcolor(LightCyan);
     settextstyle(TriplexFont, HorizDir, 3);
     settextjustify(CenterText, CenterText);
     OutTextXY(320, 30, 'Please select the pilot who you want to load:');
       {Output message (prompt)}

     new(Head);
     Head^.next := Head;
     Head^.prev := Head;
     cur := Head;
     for ctr := 1 to num do {Read in save games into linked list}
         begin {for}
              readln(IniFile, tstr); {Get the name of the save game}
              assign(SaveFile, tstr);
              reset(SaveFile); {Open save file}
              new(cur^.next);
              read(SaveFile, cur^.next^.Data); {Add data to linked list}
              close(SaveFile);
              cur^.next^.prev := cur; {Link}
              cur^.next^.next := Head;
              Head^.prev := cur^.next;
              cur := cur^.next
         end; {for}

     cur := Head^.next;
     OK := false;
     Cancel := false;
     But.init; {Init buttons}
     But.SetShadowLength(1); {Init button private constants}
     But.SetRadiusLength(1);
     But.SetTextColor(MenuTextColor);
     But.SetTextFont(SmallFont);
     But.SetTextSize(6);
     But.AddButton(240, 390, 310, 410, MenuButtonColor, '<<', '<-'); {Add menu buttons}
     But.AddButton(330, 390, 400, 410, MenuButtonColor, '>>', '->');         {|}
     But.AddButton(240, 450, 310, 470, MenuButtonColor, 'OK', 'OK');         {|}
     But.AddButton(330, 450, 400, 470, MenuButtonColor, 'CANCEL', 'CANCEL'); {|}
     But.SetTextColor(LightRed);                                             {|}
     But.AddButton(260, 420, 380, 440, LightRed, 'DELETE', 'DELETE');        {V}
     while not(OK) and not(Cancel) and not(cur = Head) do {Wait until save file selected}
           begin {while}
                ShowPilot(cur^.Data); {Show data in current element of linked list}

                But.ShowAllButtons; {Show all buttons}
                Mse.showmouse; {Show mouse cursor}
                Choice := But.Trapclick(Left_Button); {Wait for button to be chosen}
                Mse.hidemouse; {Hide mouse cursor}

                case Choice[1] of {Branch based on the button pressed}
                     '<' : begin {<-}
                                ErasePilot; {Erase pilot data}
                                cur := cur^.prev; {Move left in list}
                                if cur = Head then
                                   cur := cur^.prev
                           end; {<-}
                     '-' : begin {->}
                                ErasePilot; {Erase pilot data}
                                cur := cur^.next; {Move right in list}
                                if cur = Head then
                                   cur := cur^.next
                           end; {->}
                     'O' : begin {OK}
                                OK := true;
                                Pilot := cur^.Data {Set data to pilot}
                           end; {OK}
                     'C' : Cancel := true; {Cancel chosen}
                     'D' : begin {DELETE}
                                ErasePilot; {Erase pilot data}
                                pos := 1;
                                trav := Head^.next;
                                while trav <> cur do {Find position of save game}
                                      begin {while}
                                           pos := pos + 1;
                                           trav := trav^.next
                                      end; {while}
                                del := cur; {Delete element from linked list}
                                cur := cur^.prev;
                                cur^.next := del^.next;
                                cur^.next^.prev := cur;
                                dispose(del);
                                if cur = Head then
                                   cur := Head^.prev;
                                reset(IniFile);
                                assign(TempFile, 'TEMPINI.ALI');
                                rewrite(TempFile);
                                readln(IniFile, num);
                                writeln(TempFile, num);
                                for ctr := 1 to num do {Copy contents of .ini file to temp. file}
                                    begin {for}
                                         readln(IniFile, tstr); {Get a line}
                                         writeln(TempFile, tstr) {Put a line}
                                    end; {for}
                                readln(IniFile, num); {Get number of savegames}
                                writeln(TempFile, num - 1); {Write that number, -1 }
                                for ctr := 1 to pos - 1 do {Copy contents of .ini file to temp. file until
                                                            selected save game reached}
                                    begin {for}
                                         readln(IniFile, tstr); {Get a line}
                                         writeln(TempFile, tstr) {Put a line}
                                    end; {for}
                                readln(IniFile, tstr); {Get name of file to be deleted}
                                assign(SaveFile, tstr);
                                erase(SaveFile); {Delete it}
                                while not(eof(IniFile)) do {Copy rest of .ini file to temp. file}
                                      begin {while}
                                           read(IniFile, ch); {Get a char}
                                           write(TempFile, ch) {Put a char}
                                      end; {while}
                                close(IniFile);
                                close(TempFile);
                                reset(TempFile);
                                rewrite(IniFile);
                                while not(eof(TempFile)) do {Copy temp. file back to .ini file}
                                      begin {while}
                                           read(TempFile, ch); {Get a char}
                                           write(IniFile, ch) {Put a char}
                                      end; {while}
                                close(TempFile);
                                close(IniFile);
                                erase(TempFile) {delete temp. file}
                           end {DELETE}
                end {case}
           end; {while}

     But.Destruct; {Deallocate memory of buttons}

     if cur = Head then {If no savegames left}
        begin {if}
             dispose(Head);
             cleardevice; {Clear screen}
             setcolor(LightRed);
             settextstyle(SmallFont, HorizDir, 6);
             settextjustify(CenterText, CenterText);
             OutTextXY(320, 240, 'NO SAVED GAMES AVAILABLE!'); {Show message}
             delay(1000); {wait}
             exit {leave}
        end; {if}

     if OK then {If user chose to load}
        PlayGame(Pilot); {Play the game}
     cur := Head;
     while cur^.next <> Head do {Deallocate all memory for linked list}
           begin {while}
                del := cur^.next;
                cur^.next := del^.next;
                cur^.next^.prev := cur;
                dispose(del)
           end; {while}
     dispose(Head)
end; {LoadGame}

procedure Instructions;

{Desc: Shows instructions
 Pre:  None.
 Post: Instructions shown on screen.}  {FIX}

var
   InstFil : text; {Help.txt file}
   tstr : string; {temp. string}
   ctr : integer; {counter}

begin {Instructions}
     assign(InstFil, GetPossPath(IniPaths, 'HELP.TXT')); {Find a path for HELP.TXT}
     reset(InstFil);
     settextstyle(DefaultFont, HorizDir, 1); {Prepare font}
     settextjustify(LeftText, CenterText);
     setcolor(White);
     cleardevice; {clear screen}
     ctr := 1;
     reset(InstFil);
     while not(eof(InstFil)) do {Show all of HELP.TXT}
           begin {while}
                readln(InstFil, tstr); {Get a line from HELP.TXT}
                OutTextXY(10, ctr * 10 + 5, tstr); {Show the line on screen}
                ctr := ctr + 1;
                if ctr > 42 then {If time to show another screen}
                   begin {if}
                        setcolor(LightRed);
                        OutTextXY(10, 470, 'Press a key'); {prompt}
                        setcolor(White);
                        NormalOperation := true; {Normal keyboard operation}
                        repeat {Wait until a key is pressed}
                        until keypressed;
                        while keypressed do {Clear keyboard buffer}
                              readkey;
                        NormalOperation := false; {Re-connect keyboard interceptor}
                        cleardevice;
                        ctr := 1
                   end {if}
           end; {while}
     setcolor(LightRed);
     OutTextXY(10, 470, 'Press a key'); {Prompt}
     NormalOperation := true; {Normal keyboard operation}
     repeat {Wait until a key is pressed}
     until keypressed;
     while keypressed do {Clear keyboard buffer}
           readkey;
     NormalOperation := false; {Re-connect keyboard interceptor}
     cleardevice
end; {Instructions}

function Menu(var Pilot : Person) : boolean;

{Desc: Shows menu
 Pre:  None.
 Post: Menu put up and game started.}

var
   But : ButtonObject; {For buttons}
   Mse : MouseObject; {for mouse}
   n : integer; {number of mouse buttons}
   par : paratype; {Flashing text}
   choice : string; {Button selection}
   quit : boolean; {Flag}

begin {Menu}
     clearviewport; {Clear screen}

     setcolor(MenuBorderColor); {Draws menu}
     rectangle(200, 80, 440, 400);
     setfillstyle(SolidFill, Black);
     bar(401, 81, 439, 399);

     Mse.coldreset(n); {Cold reset mouse}
     Mse.horizlimit(200, 420); {Limit mouse movement}
     Mse.vertlimit(80, 385);
     But.init; {Init buttons}
     But.setshadowlength(1); {Length of button shadow}
     But.setradiuslength(1); {Size of button radius}
     But.settextcolor(MenuTextColor); {Color of menu text}
     But.settextfont(SmallFont); {Font of menu text}
     But.settextsize(6); {Size of font}
     But.addbutton(260, 120, 380, 150, MenuButtonColor, 'NEW GAME', 'n'); {Add buttons}
     But.addbutton(260, 180, 380, 210, MenuButtonColor, 'LOAD GAME', 'l');
     But.addbutton(260, 240, 380, 270, MenuButtonColor, 'INSTRUCTIONS', 'i');
     But.addbutton(300, 300, 340, 330, MenuButtonColor, 'QUIT', 'q');
     But.showallbuttons; {Show buttons}
     settextstyle(SmallFont, HorizDir, 5);
     settextjustify(LeftText, CenterText);
     setcolor(Yellow);
     OutTextXY(10, 469, 'Copyright (C) 1997 Po-Shen Loh   All rights reserved');
      {Show copyright info}
     settextjustify(RightText, CenterText);
     OutTextXY(630, 469, 'Version 1.72'); {shows version info}
     par[1] := 'Alien Attack!                                               '; {Prepare text}
     Fnt.psflashpara(242, 95, FlashSize, Green, 12, 20, 1, 13, FlashSpeed, 3, par); {Flash text}
     par[1] := 'By Po-Shen Loh                                              '; {Prepare text}
     Fnt.psflashpara(238, 360, FlashSize, Green, 12, 20, 1, 14, FlashSpeed, 3, par); {Flash text}
     mse.showmouse; {Show mouse cursor}

     choice := But.trapclick(Left_Button); {Get click}

     Mse.hidemouse; {Hide mouse cursor}

     quit := false;

     case choice[1] of {Branch based on button choice}
          'n' : NGame(Pilot); {Start new game}
          'l' : LoadGame(Pilot); {Load saved game}
          'i' : Instructions; {Show instructions}
          'q' : quit := true
     end; {Case}

     But.destruct; {Clear memory of buttons}

     Mse.horizlimit(1, 639); {Reset limits on mouse}
     Mse.vertlimit(1, 439);

     Menu := not(quit) {Return}
end; {Menu}

procedure SetFlags;

{Pre:  None.
 Post: All global (required) flags are reset}

var
   Shifted : byte absolute $40:$17; {Shift byte flag}

begin {SetFlags}
     clrscr;    {Clear screen}
     writeln;
     writeln('This program will start as soon as you stop touching the keyboard.');
     while keypressed or (Shifted and kbAltShift <> 0) or
           (Shifted and kbCtrlShift <> 0) or
           (Shifted and kbLeftShift <> 0) or
           (Shifted and kbRightShift <> 0) do {Wait until all hands off keyboard}
              if keypressed then {clear keyboard buffer}
                 readkey;
     clrscr; {Clear screen again}
     CtrlFlag := false;   {Reset all flags}
     UpFlag := false;
     DownFlag := false;
     LeftFlag := false;
     RightFlag := false;
     EscFlag := false;
     LShiftFlag := false;
     RShiftFlag := false;
     AltFlag := false;
     TabFlag := false
end; {SetFlags}

begin {Main}
     randomize; {Randomize generator}
     if UpCaseStr(ParamStr(1)) = '+CHEAT' then {FIX}
        TESTING := true
     else
         TESTING := false;
     mark(MemGuard); {Save memory status}
     if LookForSounds then {If sound files found}
        begin {if}
             SoundsUsed := true; {We used sounds}
             InitPSound; {Inits sound unit}
             InitSound(BaseIO, IRQ, DMA, DMA16, Sound, SoundPaths); {Inits sounds}
             if not(NoMusic) then {If there is to be background music}
                StartSound(Sound[1], 1, true)  {Starts background music}
        end {if}
     else
         SoundsUsed := false; {No sounds used}

     GetIntVec(KbdInt, OldKbdVec); {Get old keyboard interrupt vector address}

     SetIntVec(KbdInt, @Keyboard); {Set new keyboard interrupt vector to address}

     SetFlags; {Set all keyboard flags}

     Fnt.init(1.4, 5, 5, 5); {Inits graphics}
     cleardevice; {Clear screen}
     delay(2000); {Wait 2 seconds for screen to set up}

     setRGBpalette(egabrown, 15, 10, 10); {Set palette}

     GrafixOn := true; {Graphics are initialized}

     repeat {Play game until choice to quit}
     until not(Menu(Pilot));

     closegraph; {leave graphics mode}
     restoreCRTmode; {restore original screen mode}
     GrafixOn := false; {Graphics are off}

     if SoundsUsed then {If sounds in memory}
        begin {if}
             stopsound(1); {Stop music}
             shutdownsound(sound) {Free memory}
        end {if}
     else
         begin {else}
              writeln;
              writeln('Goodbye!');
              writeln;
              SetIntVec(KbdInt, OldKbdVec); {Reset keyboard vector}
              release(MemGuard) {Restore memory}
         end {else}
end. {Main}